/**************************************************************************
 *
 * lovinstem.c -- Stemming code
 * Copyright (C) 1994  Linh Huynh (linh@kbs.citri.edu.au)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include "sysfuncs.h"


/*****************************************************************************
 *
 *   Transformational Rules used in Recoding Stem Terminations
 *
 *   Parameter in functions:
 *       ch : the second last character of the old ending
 *            (after removing one of double consonants) 
 *
 ****************************************************************************/


/************ Conditional rules associated with transformations **************/

/*static int aio ();
static int s ();
static int pt ();
static int m ();
static int n ();*/

static int 
aio (char ch)			/* Rule number 9     */
{
  return ((ch != 'a') && (ch != 'i') && (ch != 'o'));
}

static int 
s (char ch)				/* Rule Number 24    */
{
  return (ch != 's');
}

static int 
pt (char ch)				/* Rule number 28    */
{
  return ((ch != 'p') && (ch != 't'));
}

static int 
m (char ch)				/* Rule number 30    */
{
  return (ch != 'm');
}

static int 
n (char ch)				/* Rule number 32    */
{
  return (ch != 'n');
}


/**************************************************************************
 *
 *  Context-sensitive rules associated with certain endings.
 *
 *  The notations of the rules are the same as in Lovins' paper
 *  except that rule A is replaced by a NULL in the data structure.
 *
 *  Function parameters:
 *      - stem_length: possible stem length
 *      - end        : pointer points to the end of the possible stem.
 *
 *************************************************************************/


/********* Context-sensitive rule function declarations ******************/

/*static int B ();
static int C ();
static int D ();
static int E ();
static int F ();
static int G ();
static int H ();
static int I ();
static int J ();
static int K ();
static int L ();
static int M ();
static int N ();
static int O ();
static int P ();
static int Q ();
static int R ();
static int S ();
static int T ();
static int U ();
static int V ();
static int W ();
static int X ();
static int Y ();
static int Z ();
static int AA ();
static int BB ();
static int CC ();
*/
static int 
B (int stem_length, char *end)
{
  return (stem_length >= 3);
}

static int 
C (int stem_length, char *end)
{
  return (stem_length >= 4);
}

static int 
D (int stem_length, char *end)
{
  return (stem_length >= 5);
}

static int 
E (int stem_length, char *end)

     
{
  return (*end != 'e');
}

static int 
F (int stem_length, char *end)
     
     
{
  return ((stem_length >= 3) && (*end != 'e'));
}

static int 
G (int stem_length, char *end)
     
     
{
  return ((stem_length >= 3) && (*end == 'f'));
}

static int 
H (int stem_length, char *end)
     
     
{
  char p1, p2;
  p1 = *end;
  p2 = *(end - 1);
  return ((p1 == 't') || ((p1 == 'l') && (p2 == 'l')));
}

static int 
I (int stem_length, char *end)
     
     
{
  char p1;
  p1 = *end;
  return ((p1 != 'o') && (p1 != 'e'));
}

static int 
J (int stem_length, char *end)
     
     
{
  char p1;
  p1 = *end;
  return ((p1 != 'a') && (p1 != 'e'));
}

static int 
K (int stem_length, char *end)
     
     
{
  char p1;
  p1 = *end;
  return ((stem_length >= 3) &&
	  ((p1 == 'l') || (p1 == 'i') ||
	   ((p1 == 'e') && (*(end - 2) == 'u'))));
}

static int 
L (int stem_length, char *end)
     
     
{
  char p1, p2;
  p1 = *end;
  p2 = *(end - 1);
  if (p1 == 's')
    return (p2 == 'o');
  else
    return ((p1 != 'u') && (p1 != 'x'));
}

static int 
M (int stem_length, char *end)
     
     
{
  char p1;
  p1 = *end;
  return ((p1 != 'a') && (p1 != 'c') &&
	  (p1 != 'e') && (p1 != 'm'));
}

static int 
N (int stem_length, char *end)
     
     
{
  if (stem_length >= 3)
    {
      if (*(end - 2) == 's')
	return (stem_length >= 4);
      else
	return 1;
    }
  else
    return 0;
}

static int 
O (int stem_length, char *end)
     
     
{
  char p1;
  p1 = *end;
  return ((p1 == 'l') || (p1 == 'i'));
}

static int 
P (int stem_length, char *end)
     
     
{
  return (*end != 'c');
}

static int 
Q (int stem_length, char *end)
     
     
{
  char p1;
  p1 = *end;

  return ((stem_length >= 3) && (p1 != 'l') && (p1 != 'n'));
}

static int 
R (int stem_length, char *end)
     
     
{
  char p1;
  p1 = *end;
  return ((p1 == 'n') || (p1 == 'r'));
}

static int 
S (int stem_length, char *end)
     
     
{
  char p1, p2;
  p1 = *end;
  p2 = *(end - 1);

  if (p1 == 't')
    return (p2 != 't');
  else
    return ((p1 == 'r') && (p2 == 'd'));
}

static int 
T (int stem_length, char *end)
     
     
{
  char p1, p2;
  p1 = *end;
  p2 = *(end - 1);

  if (p1 == 't')
    return (p2 != 'o');
  else
    return (p1 == 's');
}

static int 
U (int stem_length, char *end)
     
     
{
  char p1;
  p1 = *end;

  return ((p1 == 'l') || (p1 == 'm') ||
	  (p1 == 'n') || (p1 == 'r'));
}

static int 
V (int stem_length, char *end)
     
     
{
  return (*end == 'c');
}

static int 
W (int stem_length, char *end)
     
     
{
  char p1;
  p1 = *end;

  return ((p1 != 's') && (p1 != 'u'));
}

static int 
X (int stem_length, char *end)
     
     
{
  char p1;
  p1 = *end;

  if (p1 == 'e')
    return ((stem_length >= 3) && (*(end - 2) == 'u'));
  else
    return ((p1 == 'l') || (p1 == 'i'));
}

static int 
Y (int stem_length, char *end)
     
     
{
  return ((*end == 'n') && (*(end - 1) == 'i'));
}

static int 
Z (int stem_length, char *end)
     
     
{
  return (*end != 'f');
}

static int 
AA (int stem_length, char *end)
     
     
{
  char p1, p2;
  p1 = *end;
  p2 = *(end - 1);

  if (p1 == 'h')
    return ((p2 == 'p') || (p2 == 't'));
  else if (p1 == 'r')
    return ((p2 == 'e') || (p2 == 'o'));
  else if (p1 == 's')
    return (p2 == 'e');
  else
    return ((p1 == 'd') || (p1 == 'f') ||
	    (p1 == 'l') || (p1 == 't'));
}

static int 
BB (int stem_length, char *end)
     
     
{
  if (stem_length >= 3)
    {
      if (stem_length >= 4)
	return (strncmp (end - 3, "ryst", 4) != 0);
      else
	return (strncmp (end - 2, "met", 3) != 0);
    }
  else
    return 0;
}

static int 
CC (int stem_length, char *end)
     
     
{
  return (*end == 'l');
}

/**************************************************************************/

#define MIN_STEM_LENGTH 2
#define MAX_ENDING_SIZE 11
#define PREDEFINED_SIZE (MAX_ENDING_SIZE + MIN_STEM_LENGTH)
#define EOS     '\0'
/* [RPAP - Jan 97: Stem Index Change] */
#define MAX_STEM_LENGTH 255

static char *remove_ending (char *word, int w_len);
static void recode_stem (char *stem_end);

/* =========================================================================
 * Function: stem
 * Description: 
 *      Implemetation of the Lovins' stemming algorithm described in
 *      J.B. Lovins, "Development of a Stemming Algorithm",
 *      Mechanical Translation and Computational Linguistics, Vol 11,1968.
 * Input: a word string with the length in the first byte
 * Output: the stemmed word
 * ========================================================================= */

void
lovinstem (u_char * word)
{
  char *word_start;		/* points to first letter of word */
  char *stem_end;		/* points to last character of stem portion */
  int old_length;		/* length of word */
  int new_length;		/* length after stemming */

  /* [RPAP - Jan 97: Stem Index Change] */
  u_char copy[MAX_STEM_LENGTH + 2];
  memcpy (copy, word, *word + 1);

  old_length = *copy; /* [RPAP - Jan 97: Stem Index Change] */

  /* The word must be at least MIN_STEM_LENGTH characters long. */
  if (old_length <= MIN_STEM_LENGTH)
    return;

  word_start = (char *) copy + 1;	/* jump over length byte */ /* [RPAP - Jan 97: Stem Index Change] */
  copy[old_length + 1] = '\0';	/* null terminate string */ /* [RPAP - Jan 97: Stem Index Change] */

  stem_end = remove_ending (word_start, old_length);
  recode_stem (stem_end);

  /* fix up new length */
  /* have to use strlen because have no other way of telling length */
  new_length = strlen (word_start);
  *copy = new_length;  /* [RPAP - Jan 97: Stem Index Change] */

  /* [RPAP - Jan 97: Stem Index Change] */
  memcpy (word, copy, *copy + 1);

}				/*stem_complex */

/* =========================================================================

 *   Data structure for recoding and the method for recoding the stem. 
 *
 * ========================================================================= */

typedef struct
{
  char *old_end;		/* old ending */
  char *new_end;		/* new ending */
  char old_offset;		/* length of the old ending - 1 */
  int (*cond) (char);	        /* condition rule */
  char end_group;		/* signal the end of the group */
}
Recode_Rules;

/*  
   Stem terminations are divided into groups having the same last
   character
 */

static Recode_Rules Rules[] =
{
  {(char*)"uad", (char*)"uas", 2, NULL, 0},
  {(char*)"vad", (char*)"vas", 2, NULL, 0},
  {(char*)"cid", (char*)"cis", 2, NULL, 0},
  {(char*)"lid", (char*)"lis", 2, NULL, 0},
  {(char*)"erid", (char*)"eris", 3, NULL, 0},
  {(char*)"pand", (char*)"pans", 3, NULL, 0},
  {(char*)"end", (char*)"ens", 2, s, 0},
  {(char*)"end", (char*)"ens", 2, m, 0},
  {(char*)"ond", (char*)"ons", 2, NULL, 0},
  {(char*)"lud", (char*)"lus", 2, NULL, 0},
  {(char*)"rud", (char*)"rus", 2, NULL, 1},

  {(char*)"ul", (char*)"l", 1, aio, 1},

  {(char*)"istr", (char*)"ister", 3, NULL, 0},
  {(char*)"metr", (char*)"meter", 3, NULL, 0},
  {(char*)"her", (char*)"hes", 2, pt, 1},

  {(char*)"urs", (char*)"ur", 2, NULL, 1},

  {(char*)"uct", (char*)"uc", 2, NULL, 0},
  {(char*)"umpt", (char*)"um", 3, NULL, 0},
  {(char*)"rpt", (char*)"rb", 2, NULL, 0},
  {(char*)"mit", (char*)"mis", 2, NULL, 0},
  {(char*)"ert", (char*)"ers", 2, NULL, 0},
  {(char*)"et", (char*)"es", 1, n, 0},
  {(char*)"yt", (char*)"ys", 1, NULL, 1},

  {(char*)"iev", (char*)"ief", 2, NULL, 0},
  {(char*)"olv", (char*)"olut", 2, NULL, 1},

  {(char*)"bex", (char*)"bic", 2, NULL, 0},
  {(char*)"dex", (char*)"dic", 2, NULL, 0},
  {(char*)"pex", (char*)"pic", 2, NULL, 0},
  {(char*)"tex", (char*)"tic", 2, NULL, 0},
  {(char*)"ax", (char*)"ac", 1, NULL, 0},
  {(char*)"ex", (char*)"ec", 1, NULL, 0},
  {(char*)"ix", (char*)"ic", 1, NULL, 0},
  {(char*)"lux", (char*)"luc", 2, NULL, 1},

  {(char*)"yz", (char*)"ys", 1, NULL, 1}
};

typedef struct last
  {
    char c;			/* The last character */
    struct last *left;		/* used in balanced   */
    struct last *right;		/* binary tree        */
    Recode_Rules *pt;		/* location of approriate group */
  }
Last_Char_Node;

/*------------
       s
     /   \
    l     x
   / \   / \
  d   r t   z
         \
          v
---------------*/
static Last_Char_Node pr[] =
{
  {'d', NULL, NULL, Rules},
  {'l', pr, pr + 2, Rules + 11},
  {'r', NULL, NULL, Rules + 12},
  {'s', pr + 1, pr + 6, Rules + 15},
  {'t', NULL, pr + 5, Rules + 16},
  {'v', NULL, NULL, Rules + 23},
  {'x', pr + 4, pr + 7, Rules + 25},
  {'z', NULL, NULL, Rules + 33},
};


/*****************************************************************************
 *
 *  Recode the stem according to the transformation rules.
 *
 *****************************************************************************/

static void
recode_stem (char *stem_end)
{
  static Last_Char_Node *root = pr + 3;
  Last_Char_Node *p_last = root;
  Recode_Rules *rule;		/* points to the transformation list */
  char *h,			/* points to start of possible ending */
    ch,				/* last character of the old ending  */
    ch2;

  h = stem_end - 1;
  ch = *stem_end;
  if (*h == ch)			/* Check for double consonant        */
    {
      if (strchr ("bdglmnprst", ch) != NULL)
	{
	  *stem_end = EOS;
	  stem_end = h;
	}
    }

  do				/* Check for the last character       */
    {
      ch2 = p_last->c;
      if (ch == ch2)
	break;
      else if (ch > ch2)
	p_last = p_last->right;
      else
	p_last = p_last->left;
    }
  while (p_last != NULL);

  if (p_last != NULL)		/* Check for the rest of suffix list */
    {
      rule = p_last->pt;
      for (;;)
	{
	  h = stem_end - rule->old_offset;
	  if (strcmp (h, rule->old_end) == 0)
	    {
	      if (!rule->cond || (*rule->cond) (*(h - 1)))
		{
		  (void) strcpy (h, rule->new_end);
		  break;
		}
	    }
	  if (rule->end_group)
	    break;
	  ++rule;
	}
    }
}

/* ======================================================================

 *    List of endings and the method to remove the ending 
 *
 * ======================================================================*/

/************ Data structures for list of endings  ********************/

typedef struct
{
  char *ending;			/* old ending                */
  int (*cond) (int length, char *end);		/* conditional rule          */
  signed char left_offset;	/* used to find the siblings */
  signed char right_offset;	/* in balanced binary tree   */
}
Ending_List;

/*
   The original list of endings is re-organised into groups having
   the same length and the same first character.
 */

static Ending_List List[] =
{
  {(char*)"a", NULL, 0, 0},

  {(char*)"ae", NULL, 0, 0},
  {(char*)"al", BB, -1, 2},
  {(char*)"ar", X, 0, 0},
  {(char*)"as", B, -1, 0},

  {(char*)"acy", NULL, 0, 1},
  {(char*)"age", B, 0, 0},
  {(char*)"aic", NULL, -2, 1},
  {(char*)"als", BB, 0, 0},
  {(char*)"ant", B, -2, 2},
  {(char*)"ars", O, 0, 0},
  {(char*)"ary", F, -1, 2},
  {(char*)"ata", NULL, 0, 0},
  {(char*)"ate", NULL, -1, 0},

  {(char*)"able", NULL, 0, 1},
  {(char*)"ably", NULL, 0, 0},
  {(char*)"ages", B, -2, 2},
  {(char*)"ally", B, 0, 0},
  {(char*)"ance", B, -1, 1},
  {(char*)"ancy", B, 0, 0},
  {(char*)"ants", B, -4, 4},
  {(char*)"aric", NULL, 0, 0},
  {(char*)"arly", K, -1, 1},
  {(char*)"ated", I, 0, 0},
  {(char*)"ates", NULL, -2, 2},
  {(char*)"atic", B, 0, 0},
  {(char*)"ator", NULL, -1, 0},

  {(char*)"acies", NULL, 0, 0},
  {(char*)"acity", NULL, -1, 1},
  {(char*)"aging", B, 0, 0},
  {(char*)"aical", NULL, -2, 2},
  {(char*)"alist", NULL, 0, 0},
  {(char*)"alism", B, -1, 0},
  {(char*)"ality", NULL, -3, 3},
  {(char*)"alize", NULL, 0, 1},
  {(char*)"allic", BB, 0, 0},
  {(char*)"anced", B, -2, 2},
  {(char*)"ances", B, 0, 0},
  {(char*)"antic", C, -1, 0},
  {(char*)"arial", NULL, -6, 6},
  {(char*)"aries", NULL, 0, 1},
  {(char*)"arily", NULL, 0, 0},
  {(char*)"arity", B, -2, 2},
  {(char*)"arize", NULL, 0, 0},
  {(char*)"aroid", NULL, -1, 0},
  {(char*)"ately", NULL, -3, 3},
  {(char*)"ating", I, 0, 1},
  {(char*)"ation", B, 0, 0},
  {(char*)"ative", NULL, -2, 2},
  {(char*)"ators", NULL, 0, 0},
  {(char*)"atory", NULL, -1, 1},
  {(char*)"ature", E, 0, 0},

  {(char*)"aceous", NULL, 0, 1},
  {(char*)"acious", B, 0, 0},
  {(char*)"action", G, -2, 2},
  {(char*)"alness", NULL, 0, 0},
  {(char*)"ancial", NULL, -1, 1},
  {(char*)"ancies", NULL, 0, 0},
  {(char*)"ancing", B, -4, 4},
  {(char*)"ariser", NULL, 0, 0},
  {(char*)"arized", NULL, -1, 1},
  {(char*)"arizer", NULL, 0, 0},
  {(char*)"atable", NULL, -2, 2},
  {(char*)"ations", B, 0, 0},
  {(char*)"atives", NULL, -1, 0},

  {(char*)"ability", NULL, 0, 1},
  {(char*)"aically", NULL, 0, 0},
  {(char*)"alistic", B, -2, 2},
  {(char*)"alities", NULL, 0, 0},
  {(char*)"ariness", E, -1, 0},
  {(char*)"aristic", NULL, -3, 3},
  {(char*)"arizing", NULL, 0, 1},
  {(char*)"ateness", NULL, 0, 0},
  {(char*)"atingly", NULL, -2, 2},
  {(char*)"ational", B, 0, 0},
  {(char*)"atively", NULL, -1, 1},
  {(char*)"ativism", NULL, 0, 0},

  {(char*)"ableness", NULL, 0, 1},
  {(char*)"arizable", NULL, 0, 0},

  {(char*)"allically", C, 0, 0},
  {(char*)"antaneous", NULL, -1, 1},
  {(char*)"antiality", NULL, 0, 0},
  {(char*)"arisation", NULL, -2, 2},
  {(char*)"arization", NULL, 0, 0},
  {(char*)"ationally", B, -1, 1},
  {(char*)"ativeness", NULL, 0, 0},

  {(char*)"antialness", NULL, 0, 0},
  {(char*)"arisations", NULL, -1, 1},
  {(char*)"arizations", NULL, 0, 0},

  {(char*)"alistically", B, 0, 1},
  {(char*)"arizability", NULL, 0, 0},

  {(char*)"e", NULL, 0, 0},

  {(char*)"ed", E, 0, 0},
  {(char*)"en", F, -1, 1},
  {(char*)"es", E, 0, 0},

  {(char*)"eal", Y, 0, 0},
  {(char*)"ear", Y, -1, 1},
  {(char*)"ely", E, 0, 0},
  {(char*)"ene", E, -2, 2},
  {(char*)"ent", C, 0, 0},
  {(char*)"ery", E, -1, 1},
  {(char*)"ese", NULL, 0, 0},

  {(char*)"ealy", Y, 0, 1},
  {(char*)"edly", E, 0, 0},
  {(char*)"eful", NULL, -2, 1},
  {(char*)"eity", NULL, 0, 0},
  {(char*)"ence", NULL, -2, 2},
  {(char*)"ency", NULL, 0, 0},
  {(char*)"ened", E, -1, 2},
  {(char*)"enly", E, 0, 0},
  {(char*)"eous", NULL, -1, 0},

  {(char*)"early", Y, 0, 1},
  {(char*)"ehood", NULL, 0, 0},
  {(char*)"eless", NULL, -2, 2},
  {(char*)"elity", NULL, 0, 0},
  {(char*)"ement", NULL, -1, 0},
  {(char*)"enced", NULL, -3, 3},
  {(char*)"ences", NULL, 0, 1},
  {(char*)"eness", E, 0, 0},
  {(char*)"ening", E, -2, 2},
  {(char*)"ental", NULL, 0, 0},
  {(char*)"ented", C, -1, 1},
  {(char*)"ently", NULL, 0, 0},

  {(char*)"eature", Z, 0, 0},
  {(char*)"efully", NULL, -1, 1},
  {(char*)"encies", NULL, 0, 0},
  {(char*)"encing", NULL, -2, 2},
  {(char*)"ential", NULL, 0, 0},
  {(char*)"enting", C, -1, 1},
  {(char*)"entist", NULL, 0, 1},
  {(char*)"eously", NULL, 0, 0},

  {(char*)"elihood", E, 0, 1},
  {(char*)"encible", NULL, 0, 0},
  {(char*)"entally", NULL, -2, 2},
  {(char*)"entials", NULL, 0, 0},
  {(char*)"entiate", NULL, -1, 1},
  {(char*)"entness", NULL, 0, 0},

  {(char*)"entation", NULL, 0, 0},
  {(char*)"entially", NULL, -1, 1},
  {(char*)"eousness", NULL, 0, 0},

  {(char*)"eableness", E, 0, 1},
  {(char*)"entations", NULL, 0, 0},
  {(char*)"entiality", NULL, -2, 2},
  {(char*)"entialize", NULL, 0, 0},
  {(char*)"entiation", NULL, -1, 0},

  {(char*)"entialness", NULL, 0, 0},

  {(char*)"ful", NULL, 0, 0},

  {(char*)"fully", NULL, 0, 0},

  {(char*)"fulness", NULL, 0, 0},

  {(char*)"hood", NULL, 0, 0},

  {(char*)"i", NULL, 0, 0},

  {(char*)"ia", NULL, 0, 0},
  {(char*)"ic", NULL, -1, 1},
  {(char*)"is", NULL, 0, 0},

  {(char*)"ial", NULL, 0, 0},
  {(char*)"ian", NULL, -1, 1},
  {(char*)"ics", NULL, 0, 1},
  {(char*)"ide", L, 0, 0},
  {(char*)"ied", NULL, -3, 3},
  {(char*)"ier", NULL, 0, 0},
  {(char*)"ies", P, -1, 0},
  {(char*)"ily", NULL, -1, 1},
  {(char*)"ine", M, 0, 0},
  {(char*)"ing", N, -5, 5},
  {(char*)"ion", Q, 0, 0},
  {(char*)"ish", C, -1, 1},
  {(char*)"ism", B, 0, 1},
  {(char*)"ist", NULL, 0, 0},
  {(char*)"ite", AA, -3, 3},
  {(char*)"ity", NULL, 0, 0},
  {(char*)"ium", NULL, -1, 0},
  {(char*)"ive", NULL, -1, 1},
  {(char*)"ize", F, 0, 0},

  {(char*)"ials", NULL, 0, 0},
  {(char*)"ians", NULL, -1, 0},
  {(char*)"ible", NULL, -1, 1},
  {(char*)"ibly", NULL, 0, 0},
  {(char*)"ical", NULL, -2, 2},
  {(char*)"ides", L, 0, 0},
  {(char*)"iers", NULL, -1, 1},
  {(char*)"iful", NULL, 0, 0},
  {(char*)"ines", M, -4, 4},
  {(char*)"ings", N, 0, 0},
  {(char*)"ions", B, -1, 1},
  {(char*)"ious", NULL, 0, 0},
  {(char*)"isms", B, -2, 2},
  {(char*)"ists", NULL, 0, 0},
  {(char*)"itic", H, -1, 1},
  {(char*)"ized", F, 0, 1},
  {(char*)"izer", F, 0, 0},

  {(char*)"ially", NULL, 0, 0},
  {(char*)"icant", NULL, -1, 1},
  {(char*)"ician", NULL, 0, 0},
  {(char*)"icide", NULL, -2, 2},
  {(char*)"icism", NULL, 0, 0},
  {(char*)"icist", NULL, -1, 0},
  {(char*)"icity", NULL, -3, 3},
  {(char*)"idine", I, 0, 1},
  {(char*)"iedly", NULL, 0, 0},
  {(char*)"ihood", NULL, -2, 2},
  {(char*)"inate", NULL, 0, 0},
  {(char*)"iness", NULL, -1, 0},
  {(char*)"ingly", B, -6, 6},
  {(char*)"inism", J, 0, 1},
  {(char*)"inity", CC, 0, 0},
  {(char*)"ional", NULL, -2, 2},
  {(char*)"ioned", NULL, 0, 0},
  {(char*)"ished", NULL, -1, 0},
  {(char*)"istic", NULL, -3, 3},
  {(char*)"ities", NULL, 0, 1},
  {(char*)"itous", NULL, 0, 0},
  {(char*)"ively", NULL, -2, 2},
  {(char*)"ivity", NULL, 0, 0},
  {(char*)"izers", F, -1, 1},
  {(char*)"izing", F, 0, 0},

  {(char*)"ialist", NULL, 0, 0},
  {(char*)"iality", NULL, -1, 1},
  {(char*)"ialize", NULL, 0, 0},
  {(char*)"ically", NULL, -2, 2},
  {(char*)"icance", NULL, 0, 0},
  {(char*)"icians", NULL, -1, 1},
  {(char*)"icists", NULL, 0, 0},
  {(char*)"ifully", NULL, -4, 4},
  {(char*)"ionals", NULL, 0, 0},
  {(char*)"ionate", D, -1, 1},
  {(char*)"ioning", NULL, 0, 0},
  {(char*)"ionist", NULL, -2, 2},
  {(char*)"iously", NULL, 0, 0},
  {(char*)"istics", NULL, -1, 1},
  {(char*)"izable", E, 0, 0},

  {(char*)"ibility", NULL, 0, 0},
  {(char*)"icalism", NULL, -1, 1},
  {(char*)"icalist", NULL, 0, 1},
  {(char*)"icality", NULL, 0, 0},
  {(char*)"icalize", NULL, -3, 3},
  {(char*)"ication", G, 0, 0},
  {(char*)"icianry", NULL, -1, 0},
  {(char*)"ination", NULL, -1, 1},
  {(char*)"ingness", NULL, 0, 0},
  {(char*)"ionally", NULL, -5, 5},
  {(char*)"isation", NULL, 0, 0},
  {(char*)"ishness", NULL, -1, 1},
  {(char*)"istical", NULL, 0, 1},
  {(char*)"iteness", NULL, 0, 0},
  {(char*)"iveness", NULL, -3, 3},
  {(char*)"ivistic", NULL, 0, 0},
  {(char*)"ivities", NULL, -1, 0},
  {(char*)"ization", F, -1, 1},
  {(char*)"izement", NULL, 0, 0},

  {(char*)"ibleness", NULL, 0, 0},
  {(char*)"icalness", NULL, -1, 1},
  {(char*)"ionalism", NULL, 0, 0},
  {(char*)"ionality", NULL, -2, 2},
  {(char*)"ionalize", NULL, 0, 0},
  {(char*)"iousness", NULL, -1, 1},
  {(char*)"izations", NULL, 0, 0},

  {(char*)"ionalness", NULL, 0, 1},
  {(char*)"istically", NULL, 0, 0},
  {(char*)"itousness", NULL, -2, 2},
  {(char*)"izability", NULL, 0, 0},
  {(char*)"izational", NULL, -1, 0},

  {(char*)"izationally", B, 0, 0},

  {(char*)"ly", B, 0, 0},

  {(char*)"less", NULL, 0, 1},
  {(char*)"lily", NULL, 0, 0},

  {(char*)"lessly", NULL, 0, 0},

  {(char*)"lessness", NULL, 0, 0},

  {(char*)"ness", NULL, 0, 0},

  {(char*)"nesses", NULL, 0, 0},

  {(char*)"o", NULL, 0, 0},

  {(char*)"on", S, 0, 1},
  {(char*)"or", T, 0, 0},

  {(char*)"oid", NULL, 0, 0},
  {(char*)"one", R, -1, 1},
  {(char*)"ous", NULL, 0, 0},

  {(char*)"ogen", NULL, 0, 0},

  {(char*)"oidal", NULL, 0, 0},
  {(char*)"oides", NULL, -1, 2},
  {(char*)"otide", NULL, 0, 0},
  {(char*)"ously", NULL, -1, 0},

  {(char*)"oidism", NULL, 0, 0},

  {(char*)"oidally", NULL, 0, 1},
  {(char*)"ousness", NULL, 0, 0},

  {(char*)"s", W, 0, 0},

  {(char*)"s'", NULL, 0, 0},

  {(char*)"um", U, 0, 1},
  {(char*)"us", V, 0, 0},

  {(char*)"ward", NULL, 0, 1},
  {(char*)"wise", NULL, 0, 0},

  {(char*)"y", B, 0, 0},

  {(char*)"yl", R, 0, 0},

  {(char*)"ying", B, 0, 1},
  {(char*)"yish", NULL, 0, 0},

  {(char*)"'s", NULL, 0, 0},
};

typedef struct node
  {
    char c;			/* First character  */
    struct node *left;		/* used in balanced */
    struct node *right;		/* binary tree      */
    Ending_List *ptr[11];	/* the approriate location */
  }
First_Char_Node;

static First_Char_Node First[] =
{
  {'\'', NULL, NULL,
  {NULL,
   List + 293, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL}},

  {'a', First, NULL,
  {List,
   List + 2, List + 9, List + 20, List + 39, List + 58,
   List + 70, List + 77, List + 82, List + 87, List + 89}},

  {'e', First + 1, First + 4,
  {List + 91,
   List + 93, List + 98, List + 106, List + 116, List + 126,
   List + 133, List + 138, List + 142, List + 145, NULL}},

  {'f', NULL, NULL,
  {NULL,
   NULL, List + 146, NULL, List + 147, NULL,
   List + 148, NULL, NULL, NULL, NULL}},

  {'h', First + 3, First + 5,
  {NULL,
   NULL, NULL, List + 149, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL}},

  {'i', NULL, NULL,
  {List + 150,
   List + 152, List + 163, List + 181, List + 202, List + 222,
   List + 239, List + 252, List + 258, NULL, List + 261}},

  {'l', First + 2, First + 10,
  {NULL,
   List + 262, NULL, List + 263, NULL, List + 265,
   NULL, List + 266, NULL, NULL, NULL}},

  {'n', NULL, NULL,
  {NULL,
   NULL, NULL, List + 267, NULL, List + 268,
   NULL, NULL, NULL, NULL, NULL}},

  {'o', First + 7, First + 9,
  {List + 269,
   List + 270, List + 273, List + 275, List + 277, List + 280,
   List + 281, NULL, NULL, NULL, NULL}},

  {'s', NULL, NULL,
  {List + 283,
   List + 284, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL}},

  {'u', First + 8, First + 12,
  {NULL,
   List + 285, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL}},

  {'w', NULL, NULL,
  {NULL,
   NULL, NULL, List + 287, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL}},

  {'y', First + 11, NULL,
  {List + 289,
   List + 290, NULL, List + 291, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL}},
};


/******************************************************************************
 *
 *  Look for a match in the suffix list.
 *  Return the pointer to the end of the new stem if there is a match.
 *  Otherwise, return the pointer to the end of the original word.
 *
 *  Method:
 *      * Search for the first character in the suffix list.
 *      * If there is match, search for the rest of the suffix list.
 *         
 *****************************************************************************/

static char *
remove_ending (char *word, int w_len)
{
  static First_Char_Node *root = First + 6;	/* the root of binary tree is 'l' */
  First_Char_Node *p_first;	/* Points to the first character      */
  /* of the possible suffix             */
  Ending_List *p_list;		/* Points to the List of Endings      */
  char *suffix_start,		/* Points to start of possible suffix */
   *stem_end,			/* Points to the end of possible stem */
    ch1, ch2;
  int s_len,			/* Possible stem length               */
    e_offset,			/* Offset from the end to the start of */
  /* possible suffix                    */
    not_done = 1, cmp;

  stem_end = word + w_len - 1;
  if (w_len > PREDEFINED_SIZE)	/* find the start of suffix   */
    suffix_start = word + w_len - MAX_ENDING_SIZE;
  else
    suffix_start = word + MIN_STEM_LENGTH;

  ch1 = *suffix_start;
  do				/* Search for the first character     */
    {
      p_first = root;
      do
	{
	  ch2 = p_first->c;
	  if (ch1 == ch2)
	    break;
	  else if (ch1 > ch2)
	    p_first = p_first->right;
	  else
	    p_first = p_first->left;
	}
      while (p_first != NULL);

      if (p_first != NULL)	/* Search for the rest */
	{
	  e_offset = stem_end - suffix_start;
	  if ((p_list = p_first->ptr[e_offset]) != NULL)
	    {
	      for (;;)		/* no need to compare the first char  */
		{
		  cmp = strcmp (suffix_start + 1, p_list->ending + 1);
		  if (cmp > 0)
		    if (p_list->right_offset)
		      p_list += p_list->right_offset;
		    else
		      break;
		  else if (cmp < 0)
		    if (p_list->left_offset)
		      p_list += p_list->left_offset;
		    else
		      break;
		  else
		    {
		      s_len = suffix_start - word;
		      if (!p_list->cond ||
			  (*p_list->cond) (s_len, suffix_start - 1))
			{
			  *suffix_start = EOS;
			  stem_end = suffix_start - 1;
			  not_done = 0;
			}
		      break;
		    }
		}
	    }
	}
      ++suffix_start;
    }
  while (not_done && ((ch1 = *suffix_start) != EOS));
  return (stem_end);
}
