package org.greenstone.gatherer.feedback;

import java.util.*;
import java.awt.image.*;
import java.io.*;
import javax.imageio.*;
import javax.swing.*;
import java.awt.*;

/**
 * This class is one record of the action user did to any of the component
 * inside the window that the application open that is recorded by the 
 * special listener class.
 * An instance of this class will hold all the information about the action
 * user did at that moment,all information about all the components inside the
 * window where the action occured and all information about all the components
 * inside all the windows that were opened when the action occured.
 * @author Veronica Liesaputra
 */
public class History implements Serializable
{
    /**
     * This is the date of when the action is done by the user.
     */
    private String date;
    
    /**
     * This is the command that user did.
     */
    private String command;
    
    /**
     * This is the title of the window where the action occured.
     */
    private String title;

    /**
     * This hold all information about all the components inside the
     * window where the action occured and all information about all the components
     * inside all the windows that were open when the action occured.
     */
    private ArrayList component;

    /**
     * This hold the image of the window where the action occured.
     */
    private String image = null;

    /**
     * This is the image file name, if then user decide to save the image to a jpeg file.
     */
    private String imageName = "";

    /**
     * This is the height of image of the window where the action occured. 
     */
    private String height;

    /**
     * This is the width of image of the window where the action occured.
     */
    private String width;

    /**
     * This constructor will be initializing the date,command and component value.
     * (Precondition: (curr_date != null))
     * @param curr_date is the date when the action occured.
     * @param curr_command is the command the user did.
     */
    public History (Date curr_date,String curr_command)
    {
	date = curr_date.toString();
	command = curr_command;
	component = new ArrayList();
    }

    /**
     * This method will be setting up the image attributes to the passed values.
     * It will setting the image of the window where the action occured and its height 
     * and width.It will also generate the supposed file name for this image.
     * (Precondition: (imag != null) && (h != null) && (w != null))
     * @param img the image of the window where the action occured.
     * @param h   image's height.
     * @param w   image's width.
     */
    public void setImage(String img,String h,String w)
    {
	image = img;
	height = h;
	width = w;
	StringTokenizer st;
	st = new StringTokenizer(date,": ");
	while (st.hasMoreTokens()) 
	    {
		imageName += st.nextToken();
	    }
	imageName += ".jpg";
    }

    /**
     * This method will add a new CompGroup instance that hold information and
     * the status of a window that were open when the action occured.
     * @param comp UserComponent instance that hold information about all
     *             components inside the window.
     * @param curr_status The current status of the window.
     */
    public void addComponent (UserComponent comp,String curr_status)
    {
	component.add(new CompGroup(comp,curr_status));
    }

    /**
     * This method will get the image's height of the window where the action
     * occured.
     * @return image's height.
     */
    public String getHeight ()
    {
	return height;
    }

    /**
     * This method will get the image's width of the window where the action
     * occured.
     * @return image's width.
     */
    public String getWidth ()
    {
	return width;
    }

    /**
     * This method will set the title of the window where the action occured.
     * @param name window's title.
     */
    public void setTitle (String name)
    {
	title = name;
    }

    /**
     * This method will get the title of the window where the action occured.
     * @return the window's title.
     */
    public String getTitle ()
    {
	return title;
    }

    /**
     * This method will get the date when the action occured.
     * @return the date when the action occured.
     */
    public String getDate()
    {
	return date;
    }

    /**
     * This method will get the command that user did.
     * @return the command user did.
     */
    public String getCommand()
    {
	return command;
    }

    /**
     * This method will get the array of CompGroup instances that hold
     * all the information and status of all the window that were open when 
     * the action occured.
     * @return the array list of the CompGroup instances.
     */
    public ArrayList getArray()
    {
	return component;
    }

    /**
     * This method will get the image of the window where the action occured.
     * @return image the window's image.
     */
    public String getImage()
    {
	return image;
    }
    
    /**
     * This method will get the supposed image's file name.
     * @return image's file name.
     */
    public String getImageName()
    {
	return imageName;
    }
 
    /**
     * This method will stored this instance to an xml file.
     * It will only store the date,title,command,image and all
     * image attributes of this instance to the xml file.
     * @param sx is the xml file where we want to store.
     */
    public void sendXMLComm (SaveToXML sx)
    {
	sx.startContent(10);
	
	sx.startContent(7);
	sx.saveContent(date);
	sx.closeContent(7);
	
	sx.startContent(2);
	sx.saveContent(title);
	sx.closeContent(2);
	
	sx.startContent(8);
	sx.saveContent(command);
	sx.closeContent(8);

	if (image != null)
	    {
		sx.startContent(6);
		sx.saveImage(image,imageName,width,height);
		sx.closeContent(6);
	    }
	
	sx.closeContent(10);
    }

    /**
     * This method will stored this instance to an xml file.
     * It will store all the data members of this instance to
     * the xml file.
     * @param sx is the xml file where we want to store.
     */
    public void sendXML (SaveToXML sx)
    {
	sx.startContent(10);

	sx.startContent(7);
	sx.saveContent(date);
	sx.closeContent(7);

	sx.startContent(2);
	sx.saveContent(title);
	sx.closeContent(2);

	sx.startContent(8);
	sx.saveContent(command);
	sx.closeContent(8);

	if (image != null)
	    {
		sx.startContent(6);
		sx.saveImage(image,imageName,width,height);
		sx.closeContent(6);
	    }
	
	int j;
	for (j = 0 ; j < component.size() ; j++)
	    {
		sx.startContent(9);
		CompGroup comp;
		comp = (CompGroup) component.get(j);
		
		sx.startContent(8);
		sx.saveContent(comp.getStatus());
		sx.closeContent(8);

		sx.startContent(0);
		comp.getComponent().sendingXML(sx);
		sx.closeContent(0);
		
		sx.closeContent(9);
	    }
	sx.closeContent(10);
    }
}







