/**
 *############################################################################
 * A component of the Greenstone Librarian Interface, part of the Greenstone
 * digital library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: Michael Dewsnip, NZDL Project, University of Waikato, NZ
 *
 * Copyright (C) 2004 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *############################################################################
 */

package org.greenstone.gatherer.metadata;

import java.util.StringTokenizer;

/** This class is a static class containing useful metadata functions */
public class MetadataTools
{
    static final public String NAMESPACE_SEPARATOR = ".";
    static final public String SUBELEMENT_SEPARATOR = "^";


    static public String getDisplayNameForMetadataElementWithName(String metadata_element_name_full)
    {
	MetadataElement metadata_element = getMetadataElementWithName(metadata_element_name_full);
	if (metadata_element != null) {
	    return metadata_element.getDisplayName();
	}

	return metadata_element_name_full;
    }


    static public String getMetadataElementAttribute(MetadataElement metadata_element, String attribute_name, String language_code, String fallback_language_code)
    {
	String metadata_element_attribute = metadata_element.getAttribute(attribute_name, language_code);

	// If the attribute isn't defined in the desired language, resort to the fallback
	if (metadata_element_attribute == null && !language_code.equals(fallback_language_code)) {
	    metadata_element_attribute = metadata_element.getAttribute(attribute_name, fallback_language_code);
	}

	return metadata_element_attribute;
    }


    static public String getMetadataElementName(String metadata_element_name_full)
    {
	// Full element name contains namespace
	if (metadata_element_name_full.indexOf(NAMESPACE_SEPARATOR) != -1) {
	    return metadata_element_name_full.substring(metadata_element_name_full.indexOf(NAMESPACE_SEPARATOR) + 1);
	}

	// No namespace
	return metadata_element_name_full;      
    }


    static public MetadataElement getMetadataElementWithDisplayName(String metadata_element_display_name_full)
    {
	String metadata_set_namespace = getMetadataSetNamespace(metadata_element_display_name_full);
	MetadataSet metadata_set = MetadataSetManager.getMetadataSet(metadata_set_namespace);
	if (metadata_set == null) {
	    return null;
	}

	return metadata_set.getMetadataElementWithDisplayName(metadata_element_display_name_full);
    }


    static public MetadataElement getMetadataElementWithName(String metadata_element_name_full)
    {
	String metadata_set_namespace = getMetadataSetNamespace(metadata_element_name_full);
	MetadataSet metadata_set = MetadataSetManager.getMetadataSet(metadata_set_namespace);
	if (metadata_set == null) {
	    return null;
	}

	String metadata_element_name = getMetadataElementName(metadata_element_name_full);
	return metadata_set.getMetadataElementWithName(metadata_element_name);
    }


    static public String getMetadataSetAttribute(MetadataSet metadata_set, String attribute_name, String language_code, String fallback_language_code)
    {
	String metadata_set_attribute = metadata_set.getAttribute(attribute_name, language_code);

	// If the attribute isn't defined in the desired language, resort to the fallback
	if (metadata_set_attribute == null && !language_code.equals(fallback_language_code)) {
	    metadata_set_attribute = metadata_set.getAttribute(attribute_name, fallback_language_code);
	}

	return metadata_set_attribute;
    }


    static public String getMetadataSetNamespace(String metadata_element_name_full)
    {
	// Full element name contains namespace
	if (metadata_element_name_full.indexOf(NAMESPACE_SEPARATOR) != -1) {
	    return metadata_element_name_full.substring(0, metadata_element_name_full.indexOf(NAMESPACE_SEPARATOR));
	}

	// No namespace
	return "";
    }


    static public String getRegularExpressionThatMatchesFilePath(String file_path)
    {
	// Convert the file path into a regular expression that will match it
	String file_path_regexp = file_path;
	file_path_regexp = file_path_regexp.replaceAll("\\.", "\\\\.");
	file_path_regexp = file_path_regexp.replaceAll("\\(", "\\\\(");
	file_path_regexp = file_path_regexp.replaceAll("\\)", "\\\\)");
	file_path_regexp = file_path_regexp.replaceAll("\\[", "\\\\[");
	file_path_regexp = file_path_regexp.replaceAll("\\]", "\\\\]");
	file_path_regexp = file_path_regexp.replaceAll("\\{", "\\\\{");
	file_path_regexp = file_path_regexp.replaceAll("\\}", "\\\\}");
	file_path_regexp = file_path_regexp.replaceAll("\\+", "\\\\+");
	return file_path_regexp;
    }

    static public boolean TO_DISPLAY_NAMES = true;
    static public boolean FROM_DISPLAY_NAMES = false;

    static public String convertMetadataElementListNames(String value_str, boolean to_display) {
	if (value_str == null || value_str.length () == 0) {
	    return value_str;
	}
    
    // final true arg to return delims as tokens
    StringTokenizer string_tokenizer = new StringTokenizer (value_str, ",;/", true);
    StringBuffer value_buffer = new StringBuffer ();
    while (string_tokenizer.hasMoreElements ()) {
      String raw_token = (String) string_tokenizer.nextElement ();
      String token = raw_token.trim ();
      boolean modified_token = false;
      // not a delimiter token
      if (!raw_token.equals(",") && !raw_token.equals(";") && !raw_token.equals("/")) {
	  MetadataElement meta_elem = null;
	  if (to_display) {
	     meta_elem = getMetadataElementWithName(token);
	     if (meta_elem !=null) {
		 token = meta_elem.getDisplayName();
		 modified_token = true;
	     }
	  } else {
	      meta_elem = getMetadataElementWithDisplayName(token);
	      
	      if (meta_elem != null) {
		  token = meta_elem.getFullName ();
		  modified_token = true;
	      }
	  }
      }
      if (modified_token) {
	value_buffer.append (token);
      } else {
	// we may have had whitespace etc that was part of the string
	value_buffer.append (raw_token);
      }
    }
    
    
    if(value_buffer.length () == 0) {
      return "";
    }
    return value_buffer.toString();
    }
 

}

