/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: Katherine Don, Greenstone Digital Library, University of Waikato
 *
 * Copyright (C) 2006 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.cdm;

import org.greenstone.gatherer.util.StaticStrings;
import org.w3c.dom.*;


/** This class represents an index option. currently, might be a level or a stem option. 
 */
public class IndexOption
    implements DOMProxyListEntry {

    /** The Element this object will source its information from. */
    private Element element;

    /** Constructor used only during DOMProxyListModel initialization. */
    public IndexOption() {
    }

    /** Normal constructor.
     * @param  element the Element this object will find its data from
     */
    public IndexOption(Element element) {
	this.element = element;
    }

    /** Creation of a brand new IndexOption 
     * @param  name the name of this type as a String
     */
    public IndexOption(String name) {
	this.element = CollectionConfiguration.createElement(StaticStrings.INDEXOPTION_ELEMENT);
	this.element.setAttribute(StaticStrings.NAME_ATTRIBUTE, name);
    }

    /** Compare two objects for ordering. 
     * @param  object the other Object to compare to
     * @return <0 if this option should be before the given object, 0 if they are equal, and >0 if it should be after
     */
    public int compareTo(Object object) {
	if(object == null) {
	    return -1;
	}
	// hope we never compare a IndexOption with something that is not a IndexOption!
	return getName().compareTo(((IndexOption)object).getName());
    }

    /** Factory-like method to allow DOMProxyListModel to generate new entries.
     * @param  element the Element the new option should be based on
     * @return a newly created DOMProxyListEntry for the given element
     */
    public DOMProxyListEntry create(Element element) {
	return new IndexOption(element);
    }

    /** Determine if this option is equivalent to another object.
     * @param  object the other Object to match against
     * @return true if the two are equal, false otherwise
     */
    public boolean equals(Object object) {
	return (compareTo(object) == 0);
    }

    /** Retrieve the element this DOMProxyListEntry is based upon. Specified by the interface.
     * @return the Element in question
     */
    public Element getElement() {
	return element;
    }

    /** Retrieve the option name
     * @return the name as a String
     */
    public String getName() {
	return element.getAttribute(StaticStrings.NAME_ATTRIBUTE);
    }
    /** Retrieve the option value
     * @return the value as a string
     */
    public String getValue() {
	return element.getAttribute(StaticStrings.VALUE_ATTRIBUTE);
    }

   /** Determine is this command has been assigned, either because it already existed in the collection configuration, or because it has been explicitly set by the user. Non-assigned entries imply they have been added by the GLI to ensure consistancy (and avoid NPE's!)
     * @return true if this command has been assigned, false otherwise
     */
    public boolean isAssigned() {
	return (element != null && !element.getAttribute(StaticStrings.ASSIGNED_ATTRIBUTE).equals(StaticStrings.FALSE_STR));
    }
    
    /** Set the assigned state.
     * @param  assigned the desired state of assigned as a boolean
     */
    public void setAssigned(boolean assigned) {
	if(element != null) {
	    element.setAttribute(StaticStrings.ASSIGNED_ATTRIBUTE, (assigned ? StaticStrings.TRUE_STR : StaticStrings.FALSE_STR));
	}
    }

    /** Set the element that this DOMProxyListEntry is base on.
     * @param  element the new Element that this entry should source informatin from
     */
    public void setElement(Element element) {
	this.element = element;
    }

    /** Set the name of this option.
     * @param  name the new name for this option, as a String
     */
    public void setName(String name) {
	if(element != null) {
	    element.setAttribute(StaticStrings.NAME_ATTRIBUTE, name);
	}
    }

    /** Set the value of this option  - sometimes value is used as well as name*/
    public void setValue(String value) {
	if (element != null) {
	    element.setAttribute(StaticStrings.VALUE_ATTRIBUTE, value);
	}
    }

    /** Produce a text representation of this option
     * @return a String showing this option's name
     */
    public String toString() {
	return getName();
    }
}
