/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * <BR><BR>
 *
 * Author: John Thompson, Greenstone Digital Library, University of Waikato
 *
 * <BR><BR>
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * <BR><BR>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * <BR><BR>
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * <BR><BR>
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.gui.metaaudit;

/**************************************************************************************
 * Title:        Gatherer
 * Description:  The Gatherer: a tool for gathering and enriching a digital collection.
 * Company:      The University of Waikato
 * Written:      03/09/02
 * Revised:      04/10/02 - Commented
 **************************************************************************************/
import de.qfs.lib.gui.TableModelFilter;
import de.qfs.lib.gui.TableModelFilterEvent;
import de.qfs.lib.gui.TableModelFilterListener;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Vector;
import javax.swing.table.TableModel;

/** This class essentially manages the autofilters applied to the metaaudit table.
 * @author John Thompson
 * @version 2.3
 */
public class Filter
    implements TableModelFilter {
    /** An array of autofilters. */
    private Autofilter filters[];
    /** The registered TableModelFilterListeners. */
    private Vector listeners = new Vector ();
    /** Constructor. 
     * @param columns The number of columns this Filter will be expected to filter, as an <i>int</i>. 
     */
    public Filter(int columns) {
	filters = new Autofilter[columns];
    }
    /** Add a TableModelFilterListener to the TableModelFilter. 
     * @param listener The <strong>TableModelFilterListener</strong> to add, if it isn't already registered.
     */
    public synchronized void addTableModelFilterListener(TableModelFilterListener listener) {
	if (!listeners.contains(listener)) {
	    listeners.addElement(listener);
	}
    }
    /** Remove a filter from a column.
     * @param column The column number to remove the filter from, as an <i>int</i>.
     */
    public void clearFilter(int column) {
	if(filters[column] != null) {
	    filters[column].setActive(false);
	}
    }


    /** Determine if a certain row should be shown in the table by checking it against all the current filters.
     * @param model The <strong>TableModel</strong> the row is from.
     * @param row The row number as an <i>int</i>
     * @return <i>true</i> if the rows data matches all autofilters set and should be displayed, <i>false</i> otherwise.
     */
    public boolean filter(TableModel model, int row)
    {
	for (int i = 0; i < filters.length; i++) {
	    if (filters[i] != null && filters[i].active()) {
		ArrayList values = (ArrayList) model.getValueAt(row, i);
		if (filters[i].filter(values) == false) {
		    return false;
		}
	    }
	}

	return true;
    }


    /** Retrieve the autofilter associated with a certain column.
     * @param column The column number as an <i>int</i>.
     * @return The <strong>Autofilter</strong> assigned to that column.
     */
    public Autofilter getFilter(int column) {
	Autofilter filter = filters[column];
	if(filter == null) {
	    filter = new Autofilter();
	    filters[column] = filter;
	}
	return filter;
    }
    /** Determine if a certain column is filtered.
     * @param column The column number as an <i>int</i>.
     * @return <i>true</i> if there is an active autofilter assigned to this column, <i>false</i> otherwise.
     */
    public boolean isFiltered(int column) {
	boolean result = false;
	if(filters[column] != null) {
	    result = filters[column].active();
	}
	return result;
    }
    /** Remove a TableModelFilterListener from the TableModelFilter.
     * @param listener The <strong>TableModelFilterListener</strong> to remove.
     */
    public synchronized  void removeTableModelFilterListener(TableModelFilterListener listener) {
	listeners.removeElement(listener);
    }
    /** Called whenever the filters assigned changes significantly, thus prompting a reload of the table model data. */
    public void fireFilterChanged() {
	///ystem.err.println("Filter Changed!");
	Vector tmp;
	synchronized(this) {
	    tmp = (Vector) listeners.clone();
	}
		  
	TableModelFilterEvent event = new TableModelFilterEvent (this);
	Enumeration enumeration = tmp.elements();
	while (enumeration.hasMoreElements()) {
	    ((TableModelFilterListener) enumeration.nextElement()).filterChanged(event);
	}
    }
}
