/**
 *############################################################################
 * A component of the Greenstone Librarian Interface, part of the Greenstone
 * digital library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: Michael Dewsnip, NZDL Project, University of Waikato, NZ
 *
 * Copyright (C) 2004 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *############################################################################
 */

package org.greenstone.gatherer.metadata;


import java.io.*;
import java.util.*;
import javax.swing.tree.*;


/** This class represents all the metadata values for one metadata element */
public class MetadataValueTreeModel
    extends DefaultTreeModel
{
    public MetadataValueTreeModel(MetadataElement metadata_element)
    {
	// The root node of the value tree is the MetadataElement it represents
	super(new MetadataValueTreeNode(metadata_element));
    }


    public MetadataValueTreeNode addMetadataValue(String metadata_value)
    {
	return addMetadataValueInternal((MetadataValueTreeNode) root, metadata_value);
    }


    private MetadataValueTreeNode addMetadataValueInternal(MetadataValueTreeNode parent_metadata_value_tree_node, String metadata_value_remaining)
    {
	// Split the metadata value into a hierarchy
	int split_point = metadata_value_remaining.indexOf(MetadataValueTreeNode.METADATA_VALUE_TREE_NODE_HIERARCHY_TOKEN);
	String metadata_value = ((split_point == -1) ? metadata_value_remaining : metadata_value_remaining.substring(0, split_point));
	metadata_value_remaining = ((split_point == -1) ? "" : metadata_value_remaining.substring(split_point + 1));

	// Add the value into the tree in the correct place (sorted)
	MetadataValueTreeNode metadata_value_tree_node = null;
	for (int i = 0; i < parent_metadata_value_tree_node.getChildCount(); i++) {
	    MetadataValueTreeNode child_metadata_value_tree_node = (MetadataValueTreeNode) parent_metadata_value_tree_node.getChildAt(i);
	    int c = metadata_value.compareTo(child_metadata_value_tree_node.getValue());

	    // Insert node before existing value
	    if (c < 0) {
		metadata_value_tree_node = new MetadataValueTreeNode(metadata_value);
		insertNodeInto(metadata_value_tree_node, parent_metadata_value_tree_node, i);
		break;
	    }

	    // Node already exists (don't add again)
	    if (c == 0) {
		metadata_value_tree_node = child_metadata_value_tree_node;
		break;
	    }
	}

	// If no node has been added yet, it must go at the end of the list
	if (metadata_value_tree_node == null) {
	    metadata_value_tree_node = new MetadataValueTreeNode(metadata_value);
	    insertNodeInto(metadata_value_tree_node, parent_metadata_value_tree_node, parent_metadata_value_tree_node.getChildCount());
	}

	// If there is some of the metadata value remaining, add that hierarchically
	if (!metadata_value_remaining.equals("")) {
	    metadata_value_tree_node = addMetadataValueInternal(metadata_value_tree_node, metadata_value_remaining);
	}

	return metadata_value_tree_node;
    }


    public MetadataValueTreeNode getMetadataValueTreeNode(String metadata_value)
    {
	return getMetadataValueTreeNodeInternal((MetadataValueTreeNode) root, metadata_value);
    }


    private MetadataValueTreeNode getMetadataValueTreeNodeInternal(MetadataValueTreeNode parent_metadata_value_tree_node, String metadata_value_remaining)
    {
	// Split the metadata value into a hierarchy
	int split_point = metadata_value_remaining.indexOf(MetadataValueTreeNode.METADATA_VALUE_TREE_NODE_HIERARCHY_TOKEN);
	String metadata_value = ((split_point == -1) ? metadata_value_remaining : metadata_value_remaining.substring(0, split_point));
	metadata_value_remaining = ((split_point == -1) ? "" : metadata_value_remaining.substring(split_point + 1));

	// Find the value in the tree in the correct place (sorted)
	MetadataValueTreeNode metadata_value_tree_node = null;
	for (int i = 0; i < parent_metadata_value_tree_node.getChildCount(); i++) {
	    MetadataValueTreeNode child_metadata_value_tree_node = (MetadataValueTreeNode) parent_metadata_value_tree_node.getChildAt(i);
	    int c = metadata_value.compareTo(child_metadata_value_tree_node.getValue());

	    // Node doesn't exist in the tree
	    if (c < 0) {
		return null;
	    }
	 
	    // Node exists
	    if (c == 0) {
		metadata_value_tree_node = child_metadata_value_tree_node;
		break;
	    }
	}

	// If no node has been found, it doesn't exist
	if (metadata_value_tree_node == null) {
	    return null;
	}

	// If there is some of the metadata value remaining, find that recursively
	if (!metadata_value_remaining.equals("")) {
	    metadata_value_tree_node = getMetadataValueTreeNodeInternal(metadata_value_tree_node, metadata_value_remaining);
	}

	return metadata_value_tree_node;
    }
}
