/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: John Thompson, Greenstone Digital Library, University of Waikato
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.cdm;

import java.util.ArrayList;
import org.greenstone.gatherer.util.StaticStrings;
import org.w3c.dom.*;


/** A pretty unexciting extension of a two character string, in that it has a field which details if its the default language.
* @author John Thompson, Greenstone Digital Library, University of Waikato
* @version 2.1
*/
public class Language
    implements Comparable, DOMProxyListEntry {
    /** The Element this language entry is based upon. */
    private Element element = null;
    /** A comma separated list of two character codes */
    private String code = null;
    /** A comma separated list of language names */
    private String name = null;
 
    public Language() {
    }

    public Language(Element element) {
	this.element = element;
    }

    /** Constructor for a brand new language. */
    public Language(String code) {
	this.code = code;
	// Create the new element
	element = CollectionConfiguration.createElement(StaticStrings.LANGUAGE_ELEMENT);
	element.setAttribute(StaticStrings.NAME_ATTRIBUTE, code);
	element.setAttribute(StaticStrings.ASSIGNED_ATTRIBUTE, StaticStrings.TRUE_STR);
    }

    /** Constructor which takes an ArrayList of codes */
    public Language(ArrayList codes) {
	StringBuffer code_str = new StringBuffer();
	boolean first = true;
	for (int i=0; i<codes.size(); i++) {
	    if (!first) {
		code_str.append(",");
	    } else {
		first = false;
	    }
	    code_str.append(codes.get(i));
	}
	this.code = code_str.toString();
	element = CollectionConfiguration.createElement(StaticStrings.LANGUAGE_ELEMENT);
	element.setAttribute(StaticStrings.NAME_ATTRIBUTE, code);
	element.setAttribute(StaticStrings.ASSIGNED_ATTRIBUTE, StaticStrings.TRUE_STR);
	
    }

    /** Method to compare two languages for ordering purposes.
     * @param object the other language as an Object
     * @return an int which indicates the order between this language and the given one: < 0, 0 or > 0 for before, equal to or after respectively
     */
    public int compareTo(Object object) {
	return toString().compareTo(object.toString());
    }

    public DOMProxyListEntry create(Element element) {
	return new Language(element);
    }

    /** Method to test for the equality of two languages.
     * @param object The other language as an <strong>Object</strong>.
     * @return <i>true</i> if the languages are equal, <i>false</i> otherwise.
     */
    public boolean equals(Object object) {
	// two langs are equal if their codes are equal
	if (object instanceof Language) {
	    return getCode().equals(((Language)object).getCode());
	} else if (object instanceof String) {
	    return getCode().equals((String)object);
	}
	return false;
    }

    /** Method to retrieve the code of this language.
     * @return A <strong>String</strong> representing the two letter code.
     */
    public String getCode() {
	if(code == null && element != null) {
	    code = element.getAttribute(StaticStrings.NAME_ATTRIBUTE);
	}
	return code;
    }

    public Element getElement() {
	return element;
    }

    public String getName() {
	if(name == null) {
	    String code_str = getCode();
	    String [] codes = code_str.split(",");
	    StringBuffer name_str = new StringBuffer();
	    boolean first = true;
	    for (int i=0; i<codes.length; i++) {
		if (!first) {
		    name_str.append(",");
		} else {
		    first = false;
		}
		name_str.append(CollectionDesignManager.language_manager.getLanguageName(codes[i]));
	    }
	    name = name_str.toString();
	}
	return name;
    }

    public boolean isAssigned() {
	return (element != null && element.getAttribute(StaticStrings.ASSIGNED_ATTRIBUTE).equals(StaticStrings.TRUE_STR));
    }

    public void setAssigned(boolean value) {
	if(element != null) {
	    element.setAttribute(StaticStrings.ASSIGNED_ATTRIBUTE, (value ? StaticStrings.TRUE_STR : StaticStrings.FALSE_STR));
	}
    }

    public void setCode(String new_code) {
	code = new_code;
	// Set element
	if(element != null) {
	    element.setAttribute(StaticStrings.NAME_ATTRIBUTE, new_code);
	}
	// Reset name
	name = null;
    }

    public void setElement(Element new_element) {
	element = new_element;
	code = null;
	name = null;
    }

    /** Method to display the language code.
     * @return A <strong>String</strong> representing the language code.
     */
    public String toString() {
	return getName();
    }
}
