/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * <BR><BR>
 *
 * Author: John Thompson, Greenstone Digital Library, University of Waikato
 *
 * <BR><BR>
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * <BR><BR>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * <BR><BR>
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * <BR><BR>
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.collection;

import java.io.*;
import java.util.ArrayList;
import org.greenstone.gatherer.Configuration;
import org.greenstone.gatherer.DebugStream;
import org.greenstone.gatherer.Dictionary;
import org.greenstone.gatherer.Gatherer;
import org.greenstone.gatherer.cdm.CommandTokenizer;
import org.greenstone.gatherer.util.Codec;
import org.greenstone.gatherer.util.StaticStrings;
import org.greenstone.gatherer.util.XMLTools;
import org.w3c.dom.*;

/** This class provides access to a collection configuration file. This version accepts either a valid xml document or a historical collect.cfg file. */
public class BasicCollectionConfiguration
    implements Comparable {
    
    private File file;
    private String creator = "";
    private String description = "";
    private String maintainer = "";
    private String name = "";
    private String is_public = "";
    private String collectgroup = "false";
    
  private String short_name = "";
  
    private String site = null; // used for gs3 colls
    
    public BasicCollectionConfiguration(File file) {
	this.file = file;
	try {
	    String filename = file.getName().toLowerCase();
	    if(!file.exists()) {
		return;
	    }
	    if(filename.endsWith(".cfg")) {
		FileInputStream fis = new FileInputStream(file);
		InputStreamReader isr = new InputStreamReader(fis, "UTF-8");
		BufferedReader br = new BufferedReader(isr);
		String command = null;
		while((command = br.readLine()) != null) {
		  if (command.startsWith("#")) {
		    // a comment
		    continue;
		  }
		    if(command.length() > 0) {
			// We have to test the end of command for the special character '\'. If found, remove it and append the next line, then repeat.
			while(command.trim().endsWith("\\")) {
			    command = command.substring(0, command.lastIndexOf("\\"));
			    String next_line = br.readLine();
			    if (next_line != null) {
				command = command + "\n" + next_line;
			    }
			    next_line = null;
			}
			CommandTokenizer tokenizer = new CommandTokenizer(command, br);
			String command_type_str = tokenizer.nextToken();
			if (command_type_str == null) {
			    // Bad command. Do nothing
			    continue;
			}
			
			command_type_str = command_type_str.toLowerCase();
			if(command_type_str.equals(StaticStrings.COLLECTIONMETADATA_CREATOR_STR)) {
			    creator = tokenizer.nextToken();
			}
			else if(command_type_str.equals(StaticStrings.COLLECTIONMETADATA_MAINTAINER_STR)) {
			    maintainer = tokenizer.nextToken();
			}
			else if(command_type_str.equals(StaticStrings.COLLECTIONMETADATA_COLLECTGROUP_STR)) {
			    collectgroup = tokenizer.nextToken();
			}
			else if(command_type_str.equalsIgnoreCase(StaticStrings.COLLECTIONMETADATA_STR)) {
			    String meta_type_str = tokenizer.nextToken();
			    String value_str = tokenizer.nextToken();
			    // check for language
			    String language_str = StaticStrings.ENGLISH_LANGUAGE_STR; // assume the no-lang ones are english, but we shouldn't really do this.
			    if(meta_type_str != null && value_str != null) {
				meta_type_str = meta_type_str.toLowerCase();
				if(value_str.startsWith(StaticStrings.LBRACKET_CHARACTER) && value_str.endsWith(StaticStrings.RBRACKET_CHARACTER)) {
				    language_str = value_str.substring(value_str.indexOf(StaticStrings.EQUALS_CHARACTER) + 1, value_str.length() - 1);
				    language_str = language_str.toLowerCase();
				    value_str = tokenizer.nextToken();
				}
				// now we can work out which coll meta we are dealing with
				if (meta_type_str.equals(StaticStrings.COLLECTIONMETADATA_COLLECTIONNAME_STR)) {
				    // If this is the first collection title found, then use it, otherwise search for one that more closely matches our choosen interface language
				    if (name == null || language_str.equals(Dictionary.getLanguage())) {
					name = Codec.transform(value_str, Codec.GREENSTONE_TO_TEXT);
				    }
				}
				else if (meta_type_str.equals(StaticStrings.COLLECTIONMETADATA_COLLECTIONEXTRA_STR)) {
				    // Again we are either looking for the first description, then after that a language specific one
				    if (description == null || language_str.equals(Dictionary.getLanguage())) {
					description = Codec.transform(value_str, Codec.GREENSTONE_TO_TEXT);
				    }
				}
			    }
			    language_str = null;
			    value_str = null;
			    meta_type_str = null;
			} else {
			    // we want to process all the tokens to make sure we get rid of multi line commands before trying to find the next one
			    while (tokenizer.hasMoreTokens()) {
			    	tokenizer.nextToken();
			    }

			} // end of coll meta bit
			command_type_str = null;
			tokenizer = null;
		    } // if command.length > 0
		} // while
		command = null;
		br.close();
		isr.close();
		br = null;
		isr = null;
	    } // cfg file
	    ///ystem.err.println("Parsed collect.cfg");
	    ///ystem.err.println("name		= " + name);
	    ///ystem.err.println("creator	 = " + creator);
	    ///ystem.err.println("maintainer  = " + maintainer);
	    ///ystem.err.println("description = " + description);

	    if(filename.endsWith(".xml")) {
		
		Document xml_file_doc = XMLTools.parseXMLFile(file);
		Element root = xml_file_doc.getDocumentElement();

		Node metadataListNode = XMLTools.getChildByTagName(root, StaticStrings.METADATALIST_STR);
		Node displayItemListNode = XMLTools.getChildByTagName(root, StaticStrings.DISPLAYITEMLIST_STR);

		// Get the metadataList and displayItemList metadata for the current gli language
		String default_language = StaticStrings.ENGLISH_LANGUAGE_STR; // is this always the case?
		String gli_language_code = Configuration.getLanguage(); // definitely returns some language code

		if (metadataListNode != null) {
		  ArrayList metadata_list = XMLTools.getChildElementsByTagName((Element)metadataListNode, StaticStrings.METADATA_STR);

		  String fallback_lang_creator = "";
		  String fallback_lang_maintainer = "";
		  String fallback_lang_is_public = "";
		  
		  for (int i=0; i<metadata_list.size(); i++) {
		    Element meta = (Element) metadata_list.get(i);
		    String meta_name = meta.getAttribute(StaticStrings.NAME_ATTRIBUTE);

		    // want to get the metadata matching GLI's language, or else fall back on the default language
		    // version, or else fall back onto the first value for any language provided for the metadata
		    String meta_lang = meta.getAttribute(StaticStrings.LANG_ATTRIBUTE); // returns "" if no such attribute
		    if(meta_lang.equals("")) {
			meta_lang = default_language;
		    }
		    
		    if (meta_name.equals(StaticStrings.COLLECTIONMETADATA_CREATOR_STR)) {
			if (meta_lang.equals(gli_language_code)) { // metadata's language matched the requested language
			    creator = XMLTools.getNodeText(meta);
			} else if (meta_lang.equals(default_language)) { // meta_lang matched fall back language
			    fallback_lang_creator = XMLTools.getNodeText(meta);
			} else if (creator.equals("") && fallback_lang_creator.equals("")) { // resort to first value for meta we can find
			    fallback_lang_creator = XMLTools.getNodeText(meta);
			}
		    } else if (meta_name.equals(StaticStrings.COLLECTIONMETADATA_MAINTAINER_STR)) {
			if (meta_lang.equals(gli_language_code)) {
			    maintainer = XMLTools.getNodeText(meta);
			} else if (meta_lang.equals(default_language)) {
			    fallback_lang_maintainer = XMLTools.getNodeText(meta);
			} else if (maintainer.equals("") && fallback_lang_maintainer.equals("")) { // resort to first value for meta we can find
			    fallback_lang_maintainer = XMLTools.getNodeText(meta);
			}
		    } else if (meta_name.equals(StaticStrings.COLLECTIONMETADATA_PUBLIC_STR)) {
			if (meta_lang.equals(gli_language_code)) {
			    is_public = XMLTools.getNodeText(meta);
			} else if (meta_lang.equals(default_language)) {
			    fallback_lang_is_public = XMLTools.getNodeText(meta);
			} else if (is_public.equals("") && fallback_lang_is_public.equals("")) { // resort to first value for meta we can find
			    fallback_lang_is_public = XMLTools.getNodeText(meta);
			}
		    }		
		  }
		  // if no metadata found for the requested language, use the one set in the default language
		  if(creator.equals("")) {
		      creator = fallback_lang_creator;
		  }
		  if(maintainer.equals("")) {
		      maintainer = fallback_lang_maintainer;
		  }
		  if(is_public.equals("")) {
		      is_public = fallback_lang_is_public;
		  }
		}


		if (displayItemListNode != null) {
		  ArrayList display_list = XMLTools.getChildElementsByTagName((Element)displayItemListNode, StaticStrings.DISPLAYITEM_STR);

		  String fallback_lang_name = "";
		  String fallback_lang_description = "";

		  for (int i=0; i<display_list.size(); i++) {
		    Element meta = (Element) display_list.get(i);
		    String meta_name = meta.getAttribute(StaticStrings.NAME_ATTRIBUTE);
		    String meta_lang = meta.getAttribute(StaticStrings.LANG_ATTRIBUTE);
		    if(meta_lang.equals("")) {
			meta_lang = default_language;
		    }

		    if (meta_name.equals(StaticStrings.NAME_STR)) {
			if (meta_lang.equals(gli_language_code)) {
			    name = XMLTools.getNodeText(meta);
			} else if (meta_lang.equals(default_language)) {
			    fallback_lang_name = XMLTools.getNodeText(meta);
			} else if (name.equals("") && fallback_lang_name.equals("")) {
			    fallback_lang_name = XMLTools.getNodeText(meta);
			}
		    } else if (meta_name.equals(StaticStrings.DESCRIPTION_STR)) {
			if (meta_lang.equals(gli_language_code)) {
			    description = XMLTools.getNodeText(meta);
			} else if (meta_lang.equals(default_language)) {
			    fallback_lang_description = XMLTools.getNodeText(meta);
			} else if (description.equals("") && fallback_lang_description.equals("")) {
			    fallback_lang_description = XMLTools.getNodeText(meta);
			}
		    } 
		    
		  }

		  if(name.equals("")) {
		      name = fallback_lang_name;		      
		  }
		  if(description.equals("")) {
		      description = fallback_lang_description;
		  }
		}
	    }
	    
	}
	catch(Exception error) {
	    DebugStream.println("Error in CollectionConfiguration.<init>(): " + error);
	    DebugStream.printStackTrace(error);
	}

    }

    /** Compare this configuration to another for ordering, which essentially compares two strings for ordering.
     * @param other the other Object which is presumably another basic collection configuration
     * @return an integer which is either <0, 0 or >0 if this configuration is naturally less than, equal to or greater than the target object
     */
    public int compareTo(Object other) {
	if(other == null) {
	    return -1;
	}
	return toString().compareTo(other.toString());
    }

    public boolean equals(Object other) {
	return (compareTo(other) == 0);
    }

    /** Retrieve the creators email for this collection.
     * @return a String
     */
    public String getCreator() {
	return creator;
    }

    public String getDescription() {
	return description;
    }

    public File getFile() {
	return file;
    }
    
    public String getMaintainer() {
	return maintainer;
    }

    public String getName() {
	return name;
    }

    public String getIsPublic() {
	return is_public;
    }

    public String getCollectGroup() {
	return collectgroup;
    }
    
    /** Retrieve the short name for this collection which, given this current file is in <col_name>/etc/collect.cfg, is the name of this file's parent file's parent.
     * @return the short name of this collection as a String
     */
    public String getShortName() {
      if (short_name.equals("")) {
	String collectDir = Gatherer.getCollectDirectoryPath();
	String currentCollDir = file.getParentFile().getParentFile().getAbsolutePath(); 
	if (currentCollDir.startsWith(collectDir)) {
	  short_name = currentCollDir.substring(collectDir.length());
	  short_name = short_name.replace('\\', '/');
	} else {
	  short_name = file.getParentFile().getParentFile().getName();
	}
      }
      return short_name;
	
    }
    
    /** set the site for this coll */
    public void setSite(String site) {
	this.site = site;
    }
	
    
    /** Display the title for this collection. */
    public String toString() {
	if (this.site == null) {
	    return getName() + StaticStrings.SPACE_CHARACTER + StaticStrings.LBRACKET_CHARACTER + getShortName() + StaticStrings.RBRACKET_CHARACTER;
	} else {
	    return getName() + StaticStrings.SPACE_CHARACTER + StaticStrings.LBRACKET_CHARACTER + getShortName()+", "+this.site + StaticStrings.RBRACKET_CHARACTER;
	}
    }
}
