/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * <BR><BR>
 *
 * Author: John Thompson, Greenstone Digital Library, University of Waikato
 *
 * <BR><BR>
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * <BR><BR>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * <BR><BR>
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * <BR><BR>
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.gui;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.event.*;
import java.io.File;
import java.lang.*;
import java.net.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.filechooser.*;
import javax.swing.plaf.*;
import javax.swing.text.*;

import org.webswing.toolkit.api.WebswingUtil;

import org.greenstone.gatherer.Configuration;
import org.greenstone.gatherer.DebugStream;
import org.greenstone.gatherer.Dictionary;
import org.greenstone.gatherer.Gatherer;
import org.greenstone.gatherer.collection.Collection;
import org.greenstone.gatherer.collection.CollectionManager;
import org.greenstone.gatherer.file.FileOpenActionListener;
import org.greenstone.gatherer.file.WorkspaceTree;
import org.greenstone.gatherer.file.SingleRootFileSystemView;
import org.greenstone.gatherer.gui.metaaudit.MetaAuditFrame;
import org.greenstone.gatherer.gui.tree.DragTree;
import org.greenstone.gatherer.metadata.FilenameEncoding;
import org.greenstone.gatherer.metadata.MetadataSet;
import org.greenstone.gatherer.metadata.MetadataXMLFileManager;
import org.greenstone.gatherer.remote.RemoteGreenstoneServer;
import org.greenstone.gatherer.util.JarTools;
import org.greenstone.gatherer.util.StaticStrings;
import org.greenstone.gatherer.util.Utility;

import org.greenstone.gatherer.metadata.MetadataToCSV;
     
/** The GUIManager is in charge of creating the Gatherer window frame then filling it with the goodness of the view panes. GUIManager not only creates these panes, but allows some messaging between them. Furthermore GUIManager includes functionality from menu driven choices, simply as it was easier to put it here once and have it accessible from all pane children. */
public class GUIManager
    extends JFrame
    implements ActionListener, ChangeListener,  WindowFocusListener{
    /** The download pane contains controls for downloading internet sites. */
    public DownloadPane download_pane = null;
    /** The gather pane is more like a file manager where you drag files from one tree to another. */
    public GatherPane gather_pane = null;
    /** The enrich pane is used to assign, edit and remove metadata from files within the collection. */
    public EnrichPane enrich_pane = null;
    /** The design pane allows you to edit the design of the library in terms of the collection configuration file. - the stuff that requires rebuilding */
    public DesignPane design_pane = null;
    /** The create pane contains scripting options for importing and building collections into libraries. */
    public CreatePane create_pane = null;
    /** The format pane allows you to edit the design of the library in terms of the collection configuration file. - the stuff that doesn't require rebuilding */
    public FormatPane format_pane = null;
  public GatherPane files_pane = null;
  
    public FileOpenActionListener foa_listener = new FileOpenActionListener();

    /** A reference to the currently instantiated help window, if any. */
    private HelpFrame help = null;
    /** The menu bar. */
    public MenuBar menu_bar = null;
    public MetaAuditFrame meta_audit;
    /** Are certain panes currently locked? */
    private boolean locked = false;
    /** The size of the Gatherer window. */
    private Dimension size = null;
    /** The panel within the window that other components are placed on. */
    private JPanel content_pane = null;
    /** The last view pane selected. */
    private JPanel previous_pane;
    /** The main tab pane containing the different views, available here to trap view change events. */
    private JTabbedPane tab_pane = null;
    /** A threaded tab changer to try and avoid NPE on exit. */
    private TabUpdater tab_updater = null;

  final static String newline = "\n";
    final static String space = "    ";
   

    /**Constructor. Enable window events and arranges all other components.
     * @param size The intial <strong>Dimension</strong> of the screen.
     */
    public GUIManager(Dimension size) {
	super();
        this.setComponentOrientation(Dictionary.getOrientation());
	// Initialization
	this.help = new HelpFrame();
	this.size = size;

	if(TestingPreparation.TEST_MODE) {
	    this.setDefaultCloseOperation(DISPOSE_ON_CLOSE);
	} else {
	    this.setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
	}

	// Add a focus listener to ourselves. Thus if we gain focus when a Modal Dialog should instead have it, we can try to bring the modal dialog to the fore.
	this.addFocusListener(new GLIGUIFocusListener());

	this.addWindowFocusListener(this);
     
	// Make the Tool tip hang around for a rediculous amount of time.
	ToolTipManager.sharedInstance().setDismissDelay(10000);

	// Set up some other UI stuff. (fonts handled in Gatherer.main())
	UIManager.put("FileChooser.lookInLabelText", Dictionary.get("SaveCollectionBox.Look_In"));
	UIManager.put("FileChooser.filesOfTypeLabelText", Dictionary.get("SaveCollectionBox.Files_Of_Type"));
	UIManager.put("FileChooser.fileNameLabelText", Dictionary.get("SaveCollectionBox.File_Name"));

	// JOptionPane options
	// http://www.java2s.com/Tutorial/Java/0240__Swing/SettingJOptionPanebuttonlabelstoFrench.htm
	UIManager.put("OptionPane.cancelButtonText", Dictionary.get("General.Cancel"));
	UIManager.put("OptionPane.noButtonText", Dictionary.get("General.No"));
	UIManager.put("OptionPane.okButtonText", Dictionary.get("General.OK"));
	UIManager.put("OptionPane.yesButtonText", Dictionary.get("General.Yes"));
    }

    public void setSize(Dimension size)
    {
	this.size = size;
	super.setSize(size);
    }
    
    public void windowGainedFocus(WindowEvent e)
    {
    }

    
    public void windowLostFocus(WindowEvent e)
    {
	// Save the loaded collection
	if (Gatherer.c_man != null && Gatherer.c_man.ready() && Gatherer.c_man.getCollection().cdm != null) {
	    Gatherer.c_man.saveCollection();
	}
    }


    private class GLIGUIFocusListener
	extends FocusAdapter {
	public void focusGained(FocusEvent e) {
            if (ModalDialog.current_modal != null) {
		ModalDialog.current_modal.makeVisible();
		ModalDialog.current_modal.toFront();
	    }
	}
    }

    /** Any implementation of <i>ActionListener</i> must include this method so that we can be informed when an action has occured. In this case we are listening to actions from the menu-bar, and should react appropriately.
     * @param event An <strong>ActionEvent</strong> containing information about the action that has occured.
     */
    public void actionPerformed(ActionEvent event) {
	Object esrc = event.getSource();
	// *************
	// File Options.
	// *************
	if (esrc == menu_bar.file_associations) {
	    Gatherer.assoc_man.edit();
	}
	else if (esrc == menu_bar.file_close) {
	    saveThenCloseCurrentCollection();
	}
	else if (esrc == menu_bar.file_delete) {
	    //new DeleteCollectionTask().start();
	    SwingUtilities.invokeLater(new DeleteCollectionTask());
	}
	else if (esrc == menu_bar.file_cdimage) {
	    WriteCDImagePrompt wcdip = new WriteCDImagePrompt();
	    wcdip.display();
	    wcdip.destroy();
	    wcdip = null;
	}
	else if (esrc == menu_bar.file_exportas) {
	    ExportAsPrompt eap = new ExportAsPrompt();
	    TestingPreparation.setNamesRecursively(eap);
	    eap.display();
	    eap.destroy();
	    eap = null;
	}
	else if (esrc == menu_bar.file_exportmeta || esrc == menu_bar.file_convertmeta) {
	    String currCollName = Gatherer.c_man.getCollection().getName();
	    String collection_directory_path = CollectionManager.getCollectionDirectoryPath(currCollName);
	    
	    String importDir = collection_directory_path + "import";

	    boolean convertMode = false;
	    String startDir = collection_directory_path;
	    if(esrc == menu_bar.file_convertmeta) {
		startDir = importDir;
		convertMode = true;
	    }
		
	    File csvFile = MetadataToCSV.chooseMetaCSVFile(startDir, convertMode, this);
	    if(csvFile != null) {
		MetadataToCSV toCSV = new MetadataToCSV(collection_directory_path, csvFile);
		
		boolean success = false;
		if(esrc == menu_bar.file_convertmeta) {
		    success = toCSV.convertMetaXMLToCSV(csvFile, this);
		} else {
		    success = toCSV.exportMetaXMLToCSV(csvFile);
		}

		// refresh coll view if meta.csv was created somewhere inside this coll's import folder
		// And for remote case, also upload any new meta csv file if created anywhere inside the collection dir
		// (incl especially its import dir) to the remote server
		if(success && csvFile.exists() && csvFile.getAbsolutePath().startsWith(collection_directory_path)) {
		    // When the user *converts* metadata to CSV, it's set up to be easiest to create in the import folder
		    // When the user *exports* meta to CSV, it's set up to create this easiest in the collection folder
		    
		    // Beware in export case: client-gli can create a metadata CSV file in the collection
		    // folder and this will get successfully uploaded and store into the collect folder
		    // on the remote server. However, when the collection is reopened in client-gli, the
		    // collection that's downloaded will NOT contain such a metadata CSV file (even though it
		    // still exists on the remote server in the collection folder) since the remote server
		    // selectively sends files across when a collection is opened. 
		    // Not just anything and everything that exists in the collection folder (or its etc
		    // subfolder) is sent back to the client. There must be a reason why the GS3 server sends
		    // across some files and not others, so I'm not going to change that to incorporate any
		    // *.csv files found anywhere in the server's collection folder.
		    // When someone is using client-GLI it may not anyway make sense for them to save a file
		    // into their collection folder: the coll folder is hidden away in the gli user directory
		    // and even if client-GLI got it transferred back from the server and shows the user that
		    // their metadata.csv file exists locally, they still can't locate it that easily on
		    // their filesystem. It's best for the user to create the exported metadata in an easily
		    // located location on the client file system.
		    if(Gatherer.isGsdlRemote) {
			Gatherer.remoteGreenstoneServer.uploadFilesIntoCollection(currCollName, new File[] {csvFile}, csvFile.getParentFile());
		    }
		    if(csvFile.getAbsolutePath().startsWith(importDir)) {
			Gatherer.g_man.refreshCollectionTree(DragTree.COLLECTION_CONTENTS_CHANGED);
		    }
                    // webswing export - need to send file for download
                    if (Gatherer.isWebswing && esrc == menu_bar.file_exportmeta) {
                      // the file will be in the collection folder
                      String file_url = Configuration.library_url.toString() + "/sites/" + Configuration.site_name + "/collect/"+currCollName + "/" +csvFile.getName();

                      WebswingUtil.getWebswingApi().sendActionEvent("downloadURL", file_url, null);
                    }
		}		
	    }
	    
	}
	else if (esrc == menu_bar.file_exit) {
	    exit();
	}
        else if (esrc == menu_bar.file_new) {
	    //new NewCollectionTask().start();
	    SwingUtilities.invokeLater(new NewCollectionTask());
	}
	else if (esrc == menu_bar.file_open) {
	    //new OpenCollectionTask().start(); // will cause an EDT access violation exception
	    // since the GUI stuff of opening a collection is not done in a Swing thread
	    SwingUtilities.invokeLater(new OpenCollectionTask());
	}
	else if (esrc == menu_bar.file_options) {
	    new Preferences();
	}
	else if (esrc == menu_bar.file_save) {
	  // Very important: make sure metadata values are saved too
	  enrich_pane.stopEditingAndRebuild();
	  // Make sure all the metadata has been saved to file
	  MetadataXMLFileManager.saveMetadataXMLFiles();
	  
	  Gatherer.c_man.saveCollection();
	}

	// *************
	// Edit Options.
	// *************
	else if(esrc == menu_bar.edit_copy) {
	    try {
		KeyboardFocusManager kfm = KeyboardFocusManager.getCurrentKeyboardFocusManager();
		// Get the component with selected text as a JTextComponent
		JTextComponent text = (JTextComponent) kfm.getPermanentFocusOwner();//getFocusOwner();
		text.copy();
	    }
	    catch (Exception cce) {
		// If the component is not a text component ignore the copy command
		DebugStream.println(cce.toString());
	    }
	}
	else if(esrc == menu_bar.edit_cut) {
	    try {
		KeyboardFocusManager kfm = KeyboardFocusManager.getCurrentKeyboardFocusManager();
		// Get the component with selected text as a JTextComponent
		JTextComponent text = (JTextComponent) kfm.getPermanentFocusOwner();
		// Cut the text to the clipboard
		text.cut();
	    }
	    catch (ClassCastException cce) {
		// If the component is not a text component ignore the cut command
		DebugStream.println(cce.toString());
	    }
	}
	else if(esrc == menu_bar.edit_paste) {
	    try {
		KeyboardFocusManager kfm = KeyboardFocusManager.getCurrentKeyboardFocusManager();
		// Get the component with selected text as a JTextComponent
		JTextComponent text = (JTextComponent) kfm.getPermanentFocusOwner();
		// Cut the text to the clipboard
		text.paste();
	    }
	    catch (ClassCastException cce) {
		// If the component is not a text component ignore the paste command
		DebugStream.println(cce.toString());
	    }
	}
	else if(esrc == menu_bar.edit_config) {	   
	    if(Gatherer.c_man.getCollection() != null) {
		// Before we open the current collection's collectionConfig.xml for editing,
		// need to make sure any presently unsaved edits are saved into collConfig
		// so they are reflected in this config file when loaded into the editor for editing.
		// A simple Gatherer.c_man.saveCollection() is not sufficient: it won't save ongoing
		// edits in the currently selected GLI Pane (where relevant to colcfg)
		
		// Whenever a user swaps from one GLI pane to another, loseFocus() is called on the pane
		// and for some GLI Panes, this triggers a collection save, but notably also triggers a
		// loseFocus() on the pane's editing fields/controls, which saves any ongoing mods in that
		// pane to the collectionConfiguration file.
		// - See loseFocus() of BaseConfigPane.java (that only SOME gli Pane's inherit from)
		// and also apparent in GUIManager.windowLostFocus().
		// - And for implementations of Control.loseFocus(), see for example via
		// CollectionManager.loadDesignDetails() -> GeneralManager (for Format > General tab)
		// which leads to GeneralManager.loseFocus() which gets "Called to store the current
		// value of the components" - of its Controls. Exactly the behaviour we want!
		
		// Not all GLI Panes inherit from BaseConfigPane, but then, not all GLI Panes
		// make changes to the collection Config file; e.g. the Gather pane makes changes to 
		// the file system, Enrich makes changes to metadata.xml files and *.mds metadata sets
		// files. But Design and Format panes make changes to collectionConfig.xml.
		// In theory Create pane ought to as well, e.g. can set import options which also exists
		// as ImportOptions in collectionConfig.xml, but they're not linked (yet) and Create pane
		// doesn't inherit from BaseConfigPane. CreatePane's loseFocus() also don't call
		// for the the collection to be saved.

		// What all this means is that when we prepare to open the collectionConfig.xml,
		// we want a collection save but want also the current state of GLI edits to be saved
		// into collConfig before we display the collConfig file for editing.
		// To trigger this behaviour we want to call loseFocus() on the current pane, called
		// "previous_pane". That method may or may not force a save to collConfig,
		// depending on whether it is warranted for that pane, but in such cases it will also
		// cascade the loseFocus() call to the controls in that pane, who will save their current
		// values into the colcfg in memory before this gets saved to the colCfg.xml file.	
		
		if(previous_pane != null) {
		    doLoseFocus();
		} 
		//Gatherer.c_man.saveCollection(); //shouldn't have to do this, in theory. See above comment
                 File coll_config_file = new File(Gatherer.getCurrentCollectionDirectoryPath(),Utility.CONFIG_GS3_FILE); // col config editing is a GS3 feature

		CollectionConfigFileEditor configEditor = new CollectionConfigFileEditor(coll_config_file);
		configEditor.setVisible(true);
		configEditor.setSize(900,700);
	    }	   
	}
        // This will have only been activated if the user has permission
        else if (esrc == menu_bar.edit_group_config) {
          if (previous_pane != null) { // do we need this? editing group config doesn't affect the collection
           doLoseFocus();
          }
           String group_config_path = Gatherer.getCurrentSiteDirectoryPath()+Utility.GROUP_CONFIG_XML;
          File group_file = new File(group_config_path);
          XMLFileEditor groupConfigEditor = new XMLFileEditor(group_file);
          groupConfigEditor.setVisible(true);
          groupConfigEditor.setSize(900,700);
          
        }
 
	// *************
	// Help Options.
	// *************
	else if (esrc == menu_bar.help_general) {
	    HelpFrame.setView("introduction");
	}
	else if (esrc == menu_bar.help_download) {
	    HelpFrame.setView("themirrorview");
	}
	else if (esrc == menu_bar.help_gather) {
	    HelpFrame.setView("collectingfiles");
	}
	else if (esrc == menu_bar.help_enrich) {
	    HelpFrame.setView("enrichingacollection");
	}
	else if (esrc == menu_bar.help_design) {
	    HelpFrame.setView("designingacollection");
	}
	else if (esrc == menu_bar.help_create) {
	    HelpFrame.setView("producingthecollection");
	}
	else if (esrc == menu_bar.help_format) {
	    HelpFrame.setView("formattingacollection");
	}
	else if (esrc == menu_bar.help_about) {
	    new AboutDialog(this);
	}
    }


    /** Any actions that should happen after the display of the Gatherer window can be called here. Currently only updates the browser pane if it is active to work around bug in Mozilla renderer implementation.
     */
    public void afterDisplay() {
	if (download_pane != null) {
	    download_pane.afterDisplay();
	}
	enrich_pane.afterDisplay();
    }


    public void closeCurrentCollection()
    {
	tab_pane.setSelectedComponent(gather_pane);
	Gatherer.c_man.closeCollection();
	FilenameEncoding.closeCollection(); // clear filename-to-encodings map
    }


    public void saveThenCloseCurrentCollection()
    {
   	Gatherer.c_man.saveCollection();
	closeCurrentCollection();
    }


    /** This is called when we're absolutely finished with the GLI. It is *not* called when the applet is suspended.
     */
    public void destroy()
    {
	// Destroying create pane ensures the latest log has been closed
	if (create_pane != null) {
	    create_pane.destroy();
	}

	// Deal to help
	if (help != null) {
	    help.destroy();
	    help = null;
	}
    }


    /** Enabled events on the window to be trapped, creates all the visual components, then builds the tab and other layouts.
     */
    public void display() {
	content_pane = (JPanel) this.getContentPane();
        content_pane.setComponentOrientation(Dictionary.getOrientation());

	// Enable window-type events to be fired.
	enableEvents(AWTEvent.WINDOW_EVENT_MASK);
	// Initialise and layout sub-components, plus other window dressing.
	try {
	    this.setSize(size);

	    // Set the title
	    String collection_title = null;
	    String collection_name = null;
	    if (Gatherer.c_man.ready()) {
		Collection collection = Gatherer.c_man.getCollection();
		collection_title = collection.getTitle();
		collection_name = collection.getGroupQualifiedName(true);
		collection = null;
	    }
	    setTitle(collection_title, collection_name);
	    collection_title = null;
	    collection_name = null;

	    // Pretty corner icon
	    String gsmall_image = "gatherer.png";
	    if (Configuration.fedora_info.isActive()) {
		gsmall_image = "fli-" + gsmall_image;
	    }
		if(Gatherer.isGsdlRemote) {
	    gsmall_image = "client-" + gsmall_image;
	    }
	    this.setIconImage(JarTools.getImage(gsmall_image).getImage());
	    // BorderLayout for the main screen. I'll try my best to avoid these in subcomponents as they're space greedy.
	    content_pane.setLayout(new BorderLayout());
	    // Create the menu-bar and stick it up the top.
	    menu_bar = new MenuBar(new MenuListenerImpl());
            menu_bar.setComponentOrientation(Dictionary.getOrientation());
	    
	    //feedback changes
	    //content_pane.add(menu_bar, BorderLayout.NORTH);
	    this.setJMenuBar(menu_bar);
	    // end feedback changes

	    // Create the tabbed pane and plop it in the center where it will
	    // expand to consume all available space like any good gas would.
	    tab_pane = new JTabbedPane();
            tab_pane.setComponentOrientation(Dictionary.getOrientation());
	    tab_pane.addChangeListener(this);
	    tab_pane.setFont(Configuration.getFont("general.font", false));

	    if (Configuration.get("workflow.download", true) && Gatherer.isDownloadEnabled) {
		download_pane = new DownloadPane();
		// "GUI.Download_Tooltip" is used automatically
		tab_pane.addTab(Dictionary.get("GUI.Download"), JarTools.getImage("download.gif"), download_pane, Dictionary.get("GUI.Download_Tooltip"));
		tab_pane.setEnabledAt(tab_pane.indexOfComponent(download_pane), Configuration.get("workflow.download", false));
	    }

	    gather_pane = new GatherPane(Utility.IMPORT_MODE);          
	    gather_pane.display();
	    if (Configuration.get("workflow.gather", true)) {
		// "GUI.Gather_Tooltip" is used automatically
		tab_pane.addTab(Dictionary.get("GUI.Gather"), JarTools.getImage("gather.gif"), gather_pane, Dictionary.get("GUI.Gather_Tooltip"));
		tab_pane.setEnabledAt(tab_pane.indexOfComponent(gather_pane), Configuration.get("workflow.gather", false));
	    }

	    enrich_pane = new EnrichPane();
	    enrich_pane.display();
	    if (Configuration.get("workflow.enrich", true)) {
		// "GUI.Enrich_Tooltip" is used automatically
		tab_pane.addTab(Dictionary.get("GUI.Enrich"), JarTools.getImage("enrich.gif"), enrich_pane, Dictionary.get("GUI.Enrich_Tooltip"));
		tab_pane.setEnabledAt(tab_pane.indexOfComponent(enrich_pane), false);
	    }

	    design_pane = new DesignPane();
	    design_pane.display();
	    if (Configuration.get("workflow.design", true)) {
		// "GUI.Design_Tooltip" is used automatically
		if (Configuration.fedora_info.isActive()) {
		    tab_pane.addTab("Plugins", JarTools.getImage("design.gif"), design_pane, Dictionary.get("GUI.Design_Tooltip"));
		}
		else {
		    tab_pane.addTab(Dictionary.get("GUI.Design"), JarTools.getImage("design.gif"), design_pane, Dictionary.get("GUI.Design_Tooltip"));
		}

		tab_pane.setEnabledAt(tab_pane.indexOfComponent(design_pane), false);
	    }

	    create_pane = new CreatePane();
            create_pane.setComponentOrientation(Dictionary.getOrientation());
	    create_pane.display();
	    if (Configuration.get("workflow.create", true)) {
		// "GUI.Create_Tooltip" is used automatically
		tab_pane.addTab(Dictionary.get("GUI.Create"), JarTools.getImage("create.gif"), create_pane, Dictionary.get("GUI.Create_Tooltip"));
		tab_pane.setEnabledAt(tab_pane.indexOfComponent(create_pane), false);
	    }

	    format_pane = new FormatPane();
            format_pane.setComponentOrientation(Dictionary.getOrientation());
	    format_pane.display();
	    if (Configuration.get("workflow.format", true)) {
		tab_pane.addTab(Dictionary.get("GUI.Format"), JarTools.getImage("format.gif"), format_pane, Dictionary.get("GUI.Format_Tooltip"));
		tab_pane.setEnabledAt(tab_pane.indexOfComponent(format_pane), false);
	    }
            files_pane = new GatherPane(Utility.FILES_MODE);          
	    files_pane.display();
	    if (Configuration.get("workflow.files", true)) {
              // TODO new image
		tab_pane.addTab(Dictionary.get("GUI.Files"), JarTools.getImage("gather.gif"), files_pane, Dictionary.get("GUI.Files_Tooltip"));
		tab_pane.setEnabledAt(tab_pane.indexOfComponent(files_pane), false);
	    }

	    // The MetaAuditFrame must be created after the gather/enrich panes but before they get focus
	    meta_audit = new MetaAuditFrame();
            meta_audit.setComponentOrientation(Dictionary.getOrientation());
	    // Select the collect pane if it is available
	    if (tab_pane.indexOfComponent(gather_pane) != -1) {
		tab_pane.setSelectedComponent(gather_pane);
	    }
	    // Otherwise find the first tab that is enabled and select that.
	    else {
		for (int i = 0; i < tab_pane.getTabCount(); i++) {
		    if (tab_pane.isEnabledAt(i)) {
			tab_pane.setSelectedIndex(i);
			break;
		    }
		}
	    }

	    content_pane.add(tab_pane, BorderLayout.CENTER);

	    // Add an extra progress bar at the bottom of every screen when using a remote Greenstone server
	    if (Gatherer.isGsdlRemote) {
		JPanel remote_greenstone_server_progress_panel = new JPanel();
               //remote_greenstone_server_progress_panel.setComponentOrientation(Dictionary.getOrientation());
		JLabel remote_greenstone_server_progress_label = new JLabel(Dictionary.get("RemoteGreenstoneServer.Progress"));
                //remote_greenstone_server_progress_label.setComponentOrientation(Dictionary.getOrientation());
		remote_greenstone_server_progress_panel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
		remote_greenstone_server_progress_panel.setLayout(new BorderLayout());
		remote_greenstone_server_progress_panel.add(remote_greenstone_server_progress_label, BorderLayout.LINE_START);
		remote_greenstone_server_progress_panel.add(Gatherer.remoteGreenstoneServer.getProgressBar(), BorderLayout.CENTER);
		content_pane.add(remote_greenstone_server_progress_panel, BorderLayout.SOUTH);
	    }

	    // Call refresh to update all controls to reflect current collection status.
	    refresh(-1, Gatherer.c_man.ready());
	}
	catch (Exception e) {
	    DebugStream.printStackTrace(e);
	    // The GUI failing to build is an app killer
	    e.printStackTrace();
	    System.exit(1);
	}
    }

    public void exit() 
    {
	exit(0);
    }
	
  // some cases we have already saved the collection, but don't want to 
  // override the general.open_collection value here
  public void exitNoCollectionSave(int exit_status) {
	// Store the current position and size of the GLI for next time
	Configuration.setBounds("general.bounds", true, getBounds());

	// Save configuration
	Configuration.save();

	// Hide the main window
	setVisible(false);

	// If we're running as an applet we don't quit here (we quit when the browser calls GathererApplet.destroy())
	if (!Gatherer.isApplet) {
	    Gatherer.exit(exit_status);
	}

  }
    /** This method ensures that all the things needing saving are saved before Gatherer.exit() is called.
     */
    public void exit(int exit_status)
    {
	boolean collection_open = false;
	// If we have a collection open remember it for next time, then save it and close it
	if (Gatherer.c_man.ready()) {
		// but we no longer remember open collections for remote GLI, as this causes problems
		// in applets when other users on a machine want to use an applet and GLI wants to load
		// a collection left open from a previous session and *requires* the old user to authenticate
		if(Gatherer.isGsdlRemote) {
			Configuration.setString("general.open_collection"+Configuration.gliPropertyNameSuffix(), 
				    true, "");			
		} else {
			Configuration.setString("general.open_collection"+Configuration.gliPropertyNameSuffix(), 
				    true, CollectionManager.getLoadedCollectionColFilePath());
		}
	    saveThenCloseCurrentCollection();
		collection_open = true;
	}
	else {
	    // if there was no open collection, then write out the collect dir path for next time
		// IF this is not the default collect directory and not a remote GLI/applet
		if(Gatherer.isGsdlRemote || Gatherer.getCollectDirectoryPath().equals(
				Gatherer.getDefaultGSCollectDirectoryPath(true))) {	
			Configuration.setString("general.open_collection"+Configuration.gliPropertyNameSuffix(), 
					true, "");
		} else {
			Configuration.setString("general.open_collection"+Configuration.gliPropertyNameSuffix(), 
					true, Gatherer.getCollectDirectoryPath());
		}
	}
	
	
	// Basically, if GLI coldir != gsdlsite_colhome, we ask them if they want to save GLI coldir as gsdlsite_colhome
	// If gsdlsite_colhome = "" (meaning default GS coldir) and GLI is also set to defaultGScoldir, no saving necessary.
	// We ONLY do this for the local included apache web server, since the issue revolves around gsdlsite.cfg containing
	// the global collecthome when using the apache web server. This is not an issue for remote GS or server.exe (the latter
	// has separate config files specifying collecthome for when GLI is running whereas when just the server.exe is running).
	if(!Gatherer.isGsdlRemote && !Gatherer.isPersistentServer()) {		
	
		String defaultColDir = Gatherer.getDefaultGSCollectDirectoryPath(false); // no filesep at end
		String gliColDir = Gatherer.getCollectDirectoryPath();
		String serverColDir = Gatherer.gsdlsite_collecthome;
		
		boolean showSettingsDlg = true;
		// if collectdir was changed during GLI session (to something different from what's in gsdlsite), need to ask what to store
		String gsdlsiteColdir = Gatherer.gsdlsite_collecthome.replace("\"", "");
		if(gliColDir.equals(gsdlsiteColdir+File.separator)) {	
			showSettingsDlg = false;
		} else if(gsdlsiteColdir.equals("") // both set to default coldir
				&& gliColDir.equals(defaultColDir+File.separator)) {
			showSettingsDlg = false;
		} 
		if(showSettingsDlg) {
			
			// else we will be showing the Collect Directory Settings Dialog
			if(gliColDir.endsWith(File.separator)) {
				gliColDir = gliColDir.substring(0, gliColDir.length()-1);
			}
			
			if(serverColDir.equals("")) {
				serverColDir = defaultColDir;
			}	
			collectDirSettingsDialog(defaultColDir, serverColDir, gliColDir, collection_open);
		}
	}

	// Store the current position and size of the GLI for next time
	Configuration.setBounds("general.bounds", true, getBounds());

	// Save configuration
	Configuration.save();

	// Hide the main window
	setVisible(false);

	// If we're running as an applet we don't quit here (we quit when the browser calls GathererApplet.destroy())
	if (!Gatherer.isApplet) {
	    Gatherer.exit(exit_status);
	}
    }

	public void collectDirSettingsDialog(final String defaultColDir,
		final String from, final String to, final boolean collection_open) 
	{	
		final JDialog colHomeDialog 
			= new JDialog(this, Dictionary.get("GUI.CollectHome.title"), true); // this = Gatherer.g_man
		colHomeDialog.setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
		
		
		JRadioButton gliYes = new JRadioButton(Dictionary.get("General.Yes"), true);
						// the default selection for GLI, collecthome=to
		JRadioButton serverYes = new JRadioButton(Dictionary.get("General.Yes")); // not selected for the server		
		JRadioButton gliNo = null;
		JRadioButton gliThirdOption = null;
		JRadioButton serverNo = null;
		JRadioButton serverThirdOption = null;
		
		if(from.equals(defaultColDir)) { 
			gliNo = new JRadioButton(Dictionary.get("GUI.CollectHome.resetToDefault"));
			serverNo = new JRadioButton(Dictionary.get("GUI.CollectHome.leaveAtDefault"), true);
						// default selection for server, collecthome=from
		} else {
			gliNo = new JRadioButton(Dictionary.get("General.No"));
			serverNo = new JRadioButton(Dictionary.get("General.No"), true); // default selection for server 
			if(!to.equals(defaultColDir)) { // neither from nor to is the default GS collect dir, so give them that as the third option
				gliThirdOption = new JRadioButton(Dictionary.get("GUI.CollectHome.reset"));
				serverThirdOption = new JRadioButton(Dictionary.get("GUI.CollectHome.reset"));		
			}
		}
		
		JPanel gliPanel = new JPanel(); // flowlayout by default
		ButtonGroup gliGroup = new ButtonGroup();
		JPanel serverPanel = new JPanel(); // flowlayout by default
		ButtonGroup serverGroup = new ButtonGroup();
		
		gliGroup.add(gliYes);
		gliPanel.add(gliYes);
		serverGroup.add(serverYes);
		serverPanel.add(serverYes);		
		
		gliGroup.add(gliNo);
		gliPanel.add(gliNo);
		serverGroup.add(serverNo);
		serverPanel.add(serverNo);
			
		if(gliThirdOption != null) {
			gliThirdOption = new JRadioButton("Reset to default");
			serverThirdOption = new JRadioButton("Reset to default");
			
			gliGroup.add(gliThirdOption);
			gliPanel.add(gliThirdOption);
			serverGroup.add(serverThirdOption);
			serverPanel.add(serverThirdOption);
		}
		
		// vars need to be final to use them in the actionlistener below
		final JRadioButton gli_yes = gliYes;
		final JRadioButton gli_no = gliNo;
		final JRadioButton gli_optional = gliThirdOption;
		final JRadioButton server_yes = serverYes;
		final JRadioButton server_no = serverNo;
		final JRadioButton server_optional = serverThirdOption;
		
		JButton okButton = new JButton(Dictionary.get("General.OK"));
		okButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				// store the option chosen for GLI
				String gliColDir = to;	
				if(gli_optional != null && gli_optional.isSelected()) {
					// defaultColDir
					gliColDir = "";
				} else if(gli_yes.isSelected()) {
					gliColDir = to;
				} else if (gli_no.isSelected()) {
					gliColDir = from;
				}
				if(defaultColDir.equals(gliColDir)) {
					gliColDir = "";
				}
				if(!(collection_open && gli_yes.isSelected())) { // don't overwrite open collections
					Configuration.setString(
						"general.open_collection"+Configuration.gliPropertyNameSuffix(), 
						true, gliColDir);
				}
				
				// store the option chosen for the server's settings
				String serverColDir = from;
				if(server_optional != null && server_optional.isSelected()) {
					// defaultColDir
					serverColDir = null;
				} else if(server_yes.isSelected()) {
					serverColDir = to;
				} else if (server_no.isSelected()) {
					serverColDir = from;
				}
				if(serverColDir != null && defaultColDir.equals(serverColDir)) {					
					serverColDir = null;
				}				
				
				if(serverColDir != null) {
					serverColDir = serverColDir.replace("\"","");
					serverColDir = "\""+serverColDir+"\"";
				}
								
				Gatherer.gsdlsite_collecthome = Utility.updatePropertyConfigFile(
					Gatherer.getGsdlSiteConfigFile(), "collecthome", serverColDir);	
				
				colHomeDialog.dispose();
			}
		});
		
		//String[] dirs = {from, to};		
		//JLabel message = new JLabel(Dictionary.get("GUI.CollectHome.message", dirs));
		JLabel message = new JLabel(Dictionary.get("GUI.CollectHome.message"));
		JLabel fromDirLine = new JLabel(Dictionary.get("GUI.CollectHome.dir", from));
		fromDirLine.setBorder(BorderFactory.createEmptyBorder(0,25,0,0)); // padding
		JLabel toLine = new JLabel(Dictionary.get("GUI.CollectHome.to"));
		JLabel toDirLine = new JLabel(Dictionary.get("GUI.CollectHome.dir", to));
		toDirLine.setBorder(BorderFactory.createEmptyBorder(0,25,0,0)); // padding
		JLabel gliOption = new JLabel(Dictionary.get("GUI.CollectHome.gli"));
		JLabel serverOption = new JLabel(Dictionary.get("GUI.CollectHome.server"));		
		
		JPanel mainPanel = new JPanel();
		mainPanel.setLayout(new GridLayout(9, 1));
		mainPanel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5)); // padding
		mainPanel.add(message);
		mainPanel.add(fromDirLine);		
		mainPanel.add(toLine);
		mainPanel.add(toDirLine);
		mainPanel.add(gliOption);
		mainPanel.add(gliPanel);
		mainPanel.add(serverOption);
		mainPanel.add(serverPanel);
		mainPanel.add(okButton);		
		Container c = colHomeDialog.getContentPane();
		c.setLayout(new BorderLayout());
		c.add(mainPanel, BorderLayout.CENTER);
		colHomeDialog.getRootPane().setDefaultButton(okButton);
		
		colHomeDialog.pack();
		colHomeDialog.setVisible(true);
	}
	

    /** This method is called when the collection is being built, and is used to disable all controls in all panes which could change the state of the collection.
     */
    public void lockCollection(final boolean import_stage, final boolean lock)
    {
     Gatherer.invokeInEDT_replacesProceedInCurrThread("GUIManager.lockCollection()", Gatherer.SYNC, new Runnable() {
      public void run() {	
	locked = lock;

	if (import_stage) {
	    int gather_pos = tab_pane.indexOfComponent(gather_pane);
	    if (gather_pos != -1) {
		tab_pane.setEnabledAt(gather_pos, !lock);
	    }
	    int enrich_pos = tab_pane.indexOfComponent(enrich_pane);
	    if (enrich_pos != -1) {
		tab_pane.setEnabledAt(enrich_pos, !lock);
	    }
	}

	int design_pos = tab_pane.indexOfComponent(design_pane);
	if (design_pos != -1) {
	    tab_pane.setEnabledAt(design_pos, !lock);
	}
	int files_pos = tab_pane.indexOfComponent(files_pane);
	if (files_pos != -1) {
	    tab_pane.setEnabledAt(files_pos, !lock);
	}
      }
     });
    }


    public void modeChanged(int mode) {
	// Set the title
	String collection_title = null;
	String collection_name = null;
	if (Gatherer.c_man.ready()) {
	    Collection collection = Gatherer.c_man.getCollection();
	    collection_title = collection.getTitle();
	    collection_name = collection.getGroupQualifiedName(true);
	    collection = null;
	}
	setTitle(collection_title, collection_name);
	collection_title = null;
	collection_name = null;
	// Now pass on the message to anyone who cares
	if (download_pane != null) {
	    download_pane.modeChanged(mode);
	}
	if (gather_pane != null) {
	    gather_pane.modeChanged(mode);
	}
	if (enrich_pane != null) {
	    enrich_pane.modeChanged(mode);
	}
	if (design_pane != null) {
	    design_pane.modeChanged(mode);
	}
	if (create_pane != null) {
	    create_pane.modeChanged(mode);
	}
	if (format_pane != null) {
	    format_pane.modeChanged(mode);
	}
 	if (files_pane != null) {
	    files_pane.modeChanged(mode);
	}
        if (menu_bar != null) {
          menu_bar.modeChanged(mode);
        }
    }


    public void refresh(int refresh_reason, boolean collection_loaded)
    {
     Gatherer.invokeInEDT_replacesProceedInCurrThread("GUIManager.refresh()", Gatherer.ASYNC, new Runnable() {
      public void run() {
	// Set the collection information in the title bar
	if (collection_loaded) {
	    Collection collection = Gatherer.c_man.getCollection();
	    setTitle(collection.getTitle(), collection.getGroupQualifiedName(true));
	}
	else {
	    setTitle(null, null);
	}

	// Update the menu bar
	menu_bar.refresh(refresh_reason, collection_loaded);

	// Update the loaded panes
	if (download_pane != null) {
	    download_pane.refresh(refresh_reason, collection_loaded);
	}
	if (gather_pane != null) {
	    gather_pane.refresh(refresh_reason, collection_loaded);
	}
	if (enrich_pane != null) {
	    enrich_pane.refresh(refresh_reason, collection_loaded);
	}
	if (design_pane != null) {
	    design_pane.refresh(refresh_reason, collection_loaded);
	}
	if (create_pane != null) {
	    create_pane.refresh(refresh_reason, collection_loaded);
	}
	if (format_pane != null) {
	    format_pane.refresh(refresh_reason, collection_loaded);
	}
	if (files_pane != null) {
	    files_pane.refresh(refresh_reason, collection_loaded);
	}
      }
	 }); // end Gatherer.invokeInEDT()
	
	// Now enable tabs as necessary. Do this on event queue to prevent crazy NPEs
	if (!locked) {
	    if (tab_updater == null) {
		tab_updater = new TabUpdater(tab_pane, collection_loaded);
	    }
	    else {
		tab_updater.setReady(collection_loaded);
	    }
	    SwingUtilities.invokeLater(tab_updater);
	}
    }


    public void refreshCollectionTree(int refresh_reason)
    {
	if (gather_pane != null) {
	    gather_pane.refreshCollectionTree(refresh_reason);
	}
        if (files_pane != null) {
          files_pane.refreshCollectionTree(refresh_reason);
        }
    }


    public void refreshWorkspaceTree(int refresh_reason)
    {
	if (gather_pane != null) {
	    gather_pane.refreshWorkspaceTree(refresh_reason);
	}
	if (files_pane != null) {
	    files_pane.refreshWorkspaceTree(refresh_reason);
	}
    }

    public void refreshWorkspaceTreeGreenstoneCollections()
    {
	refreshWorkspaceTree(WorkspaceTree.LIBRARY_CONTENTS_CHANGED);
    }

  public boolean isGatherPaneSelected() {
    int index = tab_pane.getSelectedIndex();
    int gather_index = tab_pane.indexOfTab(Dictionary.get("GUI.Gather"));
    if (index == gather_index) {
      return true;
    }
    return false;
  }

    /** Specifies whether a certain tab is enabled or not. */
    private void setTabEnabled(String rawname, boolean state) {
	// Retrieve the dictionary based name.
	String name = Dictionary.get("GUI." + rawname);
	int index = tab_pane.indexOfTab(name);
	// Of course we may not have this tab available.
	if(index != -1) {
	    // Some tabs are also dependant on if a collection is ready
	    Component component = tab_pane.getComponentAt(index);
	    if(component == enrich_pane || component == design_pane || component == create_pane || component == format_pane || component == files_pane) {
		tab_pane.setEnabledAt(index, state && Gatherer.c_man != null && Gatherer.c_man.ready());
	    }
	    else {
		tab_pane.setEnabledAt(index, state);
	    }
	    // If this was the currently selected tab and it is now disabled, change the view to the first enabled tab.
	    if(tab_pane.getSelectedIndex() == index && !state) {
		boolean found = false;
		for(int i = 0; !found && i < tab_pane.getTabCount(); i++) {
		    if(tab_pane.isEnabledAt(i)) {
			tab_pane.setSelectedIndex(i);
			found = true;
		    }
		}
		// If there are no tabs enabled, which should be impossible, then select the first tab
		if(!found) {
		    tab_pane.setSelectedIndex(0);
		}
	    }
	}
    }

    /** Change the string shown in the title bar of the main gui frame. If either value is null, the 'No Collection' string is shown instead.
     * @param title
     * @param name
     */
    public void setTitle(String title, String name) {
      
	// Finally display the collection name in the title bar.
	StringBuffer title_buffer = new StringBuffer(Configuration.getApplicationTitle());
	title_buffer.append(StaticStrings.SPACE_CHARACTER);
	title_buffer.append(Gatherer.PROGRAM_VERSION);
	title_buffer.append(StaticStrings.SPACE_CHARACTER);
	title_buffer.append(StaticStrings.SPACE_CHARACTER);
	// Server version information
	title_buffer.append(Gatherer.getServerVersionAsString());
	title_buffer.append(StaticStrings.SPACE_CHARACTER);
	title_buffer.append(StaticStrings.SPACE_CHARACTER);	
	// Describe the current user mode
	title_buffer.append(StaticStrings.MODE_STR);
	title_buffer.append(Configuration.getModeAsString());
	title_buffer.append(StaticStrings.SPACE_CHARACTER);
	title_buffer.append(StaticStrings.SPACE_CHARACTER);
	// Now for the current collection
	title_buffer.append(StaticStrings.COLLECTION_STR);
	if (title != null && name != null) {
	    title_buffer.append(title);
	    title_buffer.append(StaticStrings.SPACE_CHARACTER);
	    title_buffer.append(StaticStrings.LBRACKET_CHARACTER);
	    title_buffer.append(name);
	    title_buffer.append(StaticStrings.RBRACKET_CHARACTER);
	}
	else {
	    title_buffer.append(Dictionary.get("Collection.No_Collection"));
	}
	this.setTitle(title_buffer.toString());
	title_buffer = null;
    }


    private class OpenCollectionTask
    implements Runnable //extends Thread -> If this extends Thread, it will cause an EDT access
	       // violation since the GUI stuff in run() is not done in the Swing Event Dispatch Thread/EDT.
    {
	public void run()
	{
	    String collection_file_path = showOpenCollectionDialog();

	    // User has selected a collection to open
	    if (collection_file_path != null) {
		// If there is already a collection open, save and close it
		if (Gatherer.c_man.ready()) {
		    saveThenCloseCurrentCollection();
		}

		// Open the selected collection
		Gatherer.c_man.loadCollection(collection_file_path);
	    }
	}
    }


    /** When the load collection option is chosen this method is called to produce the modal file load prompt.
     */
    private String showOpenCollectionDialog()
    {
	OpenCollectionDialog dialog = new OpenCollectionDialog();
	TestingPreparation.setNamesRecursively(dialog);
	
        dialog.setComponentOrientation(Dictionary.getOrientation());
	if (dialog.display() == OpenCollectionDialog.OK_OPTION) {
	    return dialog.getFileName();
	}

	// User must have cancelled the action
	return null;
    }


    /** When called this method causes the MetadataAuditTable to display a nice dialog box which contains all the metadata assigned in the collection.
     */
    public void showMetaAuditBox() {
	wait(true);
	meta_audit.display();
	wait(false);
    }


    private class NewCollectionTask
	implements Runnable //extends Thread
    {
	public void run()
	{
	    // Create the collection details prompt from new collection prompt
	    NewCollectionDetailsPrompt ncd_prompt = new NewCollectionDetailsPrompt();	    
	    //TestingPreparation.setNamesRecursively(ncd_prompt);
	    // 1. names have to be set in NewCollectionDetailsPrompt, as it makes itself visible
	    // and for testing we have to have run setName() on its components before the dialog goes visible
	    // 2. For whatever reason, the NewCollectionDetailsPrompt dialog's own name is set
	    // to "col" and it's not possible to change it at any point before or after
	    // creating the dialog, and whether from inside or outside the dialog's own class

	    // Create the new collection (if not cancelled) in a new thread.
	    if (!ncd_prompt.isCancelled()) {
		// If there is already a collection open, save and close it.
		if (Gatherer.c_man.ready()) {
		    saveThenCloseCurrentCollection();
		}
		
		// Create new collection.
		Gatherer.c_man.createCollection(ncd_prompt.getDescription(), Configuration.getEmail(), ncd_prompt.getName(), ncd_prompt.getTitle(), ncd_prompt.getBase(), new ArrayList());
		ncd_prompt.dispose();
	    }

	    // Done
	    ncd_prompt = null;
	}
    }


    private class DeleteCollectionTask
    implements Runnable //extends Thread
    {
	public void run()
	{
	    // The rest is handled by the DeleteCollectionPrompt
	    DeleteCollectionPrompt dc_prompt = new DeleteCollectionPrompt();
	    TestingPreparation.setNamesRecursively(dc_prompt);
	    if (dc_prompt.display()) {
		//closeCurrentCollection();
	    }
	    dc_prompt.destroy();
	    dc_prompt = null;
	}
    }

    // Want to triger loseFocus() on GLI Panes at will when Menu > Edit > collectionConfig.xml clicked.
    // And when done editing, want to trigger doRegainFocs().
    // loseFocus() on GLI Panes is not universally inherited from a common base class
    // But loseFocus() is important as it forces saves as-is of user entered values in gui fields
    // When alling doLoseFocus() on GLI > Edit > collConfig.xml, we force a save of the current state
    // of the current pane (called "previous_pane").
    // Else a save is only triggered when another GLI Pane is clicked. We can't expect a user to
    // remember to click other panes whenever they want to edit the collConfig.xml.    
    public void doLoseFocus() {
	if (previous_pane != null) {
	    if (previous_pane == gather_pane) {
		gather_pane.loseFocus();
	    }
	    else if (previous_pane == enrich_pane) {
		enrich_pane.loseFocus();
	    }
	    else if (previous_pane == design_pane) {
		design_pane.loseFocus();
	    }
	    else if (previous_pane == create_pane) {
		create_pane.loseFocus();
	    }
	    else if (previous_pane == format_pane) {
		format_pane.loseFocus();
	    }
	    else if (previous_pane == files_pane) {
		files_pane.loseFocus();
	    }
	}
    }
    public void doRegainFocus() {
	if (previous_pane != null) {
	    if (previous_pane == gather_pane) {
		gather_pane.gainFocus();
	    }
	    else if (previous_pane == enrich_pane) {
		enrich_pane.gainFocus();
	    }
	    else if (previous_pane == design_pane) {
		design_pane.gainFocus();
	    }
	    else if (previous_pane == create_pane) {
		create_pane.gainFocus();
	    }
	    else if (previous_pane == format_pane) {
		format_pane.gainFocus();
	    }
	    else if (previous_pane == files_pane) {
		files_pane.gainFocus();
	    }
	}
    }
    
    /** Any implementation of ChangeListener must include this method so we can be informed when the state of one of the registered objects changes. In this case we are listening to view changes within the tabbed pane.
     * @param event A ChangeEvent containing information about the event that fired this call.
     */
    public void stateChanged(ChangeEvent event)
    {
	if (previous_pane != null) {
	    if (previous_pane == gather_pane) {
		gather_pane.loseFocus();
	    }
	    else if (previous_pane == enrich_pane) {
		enrich_pane.loseFocus();
	    }
	    else if (previous_pane == design_pane) {
		design_pane.loseFocus();
	    }
	    else if (previous_pane == create_pane) {
		create_pane.loseFocus();
	    }
	    else if (previous_pane == format_pane) {
		format_pane.loseFocus();
	    }
	    else if (previous_pane == files_pane) {
		files_pane.loseFocus();
	    }
	}

	menu_bar.tabSelected(tab_pane.getSelectedIndex());
	int selected_index = tab_pane.getSelectedIndex();
	if (selected_index == tab_pane.indexOfComponent(download_pane)) {
	    download_pane.gainFocus();
	}
	else if (selected_index == tab_pane.indexOfComponent(gather_pane)) {
	    gather_pane.gainFocus();
	}
	else if (selected_index == tab_pane.indexOfComponent(enrich_pane)) {
	    enrich_pane.gainFocus();
	}
	else if (selected_index == tab_pane.indexOfComponent(design_pane)) {
	    design_pane.gainFocus();
	}
	else if (selected_index == tab_pane.indexOfComponent(create_pane)) {
	    create_pane.gainFocus();
	}
	else if (selected_index == tab_pane.indexOfComponent(format_pane)) {
	    format_pane.gainFocus();
	}
	else if (selected_index == tab_pane.indexOfComponent(files_pane)) {
	    files_pane.gainFocus();
	}

	previous_pane = (JPanel) tab_pane.getSelectedComponent();
	
    }


    private MouseListener mouse_blocker_listener = new MouseAdapter() {};

    public void updateUI()
    {
	SwingUtilities.invokeLater(new Runnable() {
		public void run() {
		    JPanel pane = (JPanel) getContentPane();
		    pane.updateUI();
		    // Also update all of the tabs according to workflow.
		    workflowUpdate("Download", Configuration.get("workflow.download", false));
		    workflowUpdate("Gather", Configuration.get("workflow.gather", false));
		    workflowUpdate("Enrich", Configuration.get("workflow.enrich", false));
		    workflowUpdate("Design", Configuration.get("workflow.design", false));
		    workflowUpdate("Create", Configuration.get("workflow.create", false));
		    workflowUpdate("Format", Configuration.get("workflow.format", false));
		    workflowUpdate("Files", Configuration.get("workflow.files", false));
	    	}
	    });
    }

    public void wait(boolean waiting) {
     Gatherer.invokeInEDT_replacesProceedInCurrThread(
      "GUIManager.wait()",
      Gatherer.SYNC,
      //Gatherer.BYPASS_WHEN_NOT_TESTING,
      new Runnable() {
       public void run() {
		 
	Component glass_pane = getGlassPane();
	if(waiting) {
	    // Show wait cursor.
	    glass_pane.addMouseListener(mouse_blocker_listener);
	    glass_pane.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
	    glass_pane.setVisible(true);
	}
	else {
	    // Hide wait cursor.
	    glass_pane.setVisible(false);
	    glass_pane.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
	    glass_pane.removeMouseListener(mouse_blocker_listener);
	}
	glass_pane = null;
	
       } // end run()
      }); // end invokeInEDT()
	
    }

    public void workflowUpdate(String raw, boolean state) {
	WorkflowUpdater task = new WorkflowUpdater(raw, state);
	SwingUtilities.invokeLater(task);
	task = null;
    }


    /**Overridden from JFrame so we can exit safely when window is closed (or destroyed).
     * @param event A <strong>WindowEvent</strong> containing information about the event that fired this call.
     */
    protected void processWindowEvent(WindowEvent event) {
	if(event.getID() == WindowEvent.WINDOW_CLOSING) {
	    exit();
	}
    }


    /** Listens to actions upon the menu bar, and if it detects a click over the help menu brings the help window to the front if it has become hidden.
     */
    private class MenuListenerImpl
	implements MenuListener {
	/** Called whenever a popup menu is hidden, but we don't care.
	 * @param e Some <strong>MenuEvent</strong> that we could care less about.
	 */
	public void menuCanceled(MenuEvent e) {
	}
	/** Called whenever a menu header (ie button) becomes unselected, but we don't care.
	 * @param e Some <strong>MenuEvent</strong> that we could care less about.
	 */
	public void menuDeselected(MenuEvent e) {
	}
	/** This method, when a menu is first opened, is the only one we respond to by bringing the help window to the front if possible, but only if there is a help window and the help menu is the one opening.
	 * @param e The <strong>MenuEvent</strong> whose source is checked.
	 */
	public void menuSelected(MenuEvent e) {
	    if(e.getSource() == menu_bar.help) {
		if(menu_bar.help.isSelected()) {
		    menu_bar.help.doClick(10);
		}
	    }
	    else if(e.getSource() == menu_bar.edit) { // someone clicked the toplevel Edit menu
		// gray out the Edit Config File menu if there's no collection currently open.
		if(Gatherer.c_man.getCollection() == null) {
		    menu_bar.edit_config.setEnabled(false);
		}
		else { // don't forget to reenable the Edit ConfigXML menu item when applicable
		    menu_bar.edit_config.setEnabled(true);
		}
	    }

	}
    }

    private class TabUpdater
	implements Runnable {
	private boolean ready = false;
	private int download_pos = -1;
	private int enrich_pos = -1;
	private int design_pos = -1;
	private int create_pos = -1;
	private int format_pos = -1;
      private int files_pos = -1;
	private int export_pos = -1;
	private JTabbedPane tab_pane = null;

	public TabUpdater(JTabbedPane tab_pane, boolean ready) {
	    this.ready = ready;
	    this.tab_pane = tab_pane;
	    download_pos = tab_pane.indexOfComponent(download_pane);
	    enrich_pos = tab_pane.indexOfComponent(enrich_pane);
	    design_pos = tab_pane.indexOfComponent(design_pane);
	    create_pos = tab_pane.indexOfComponent(create_pane);
	    format_pos = tab_pane.indexOfComponent(format_pane);
	    files_pos = tab_pane.indexOfComponent(files_pane);
	}

	public void run()
	{
	    if (download_pos != -1) {
		if (ready) {
		    tab_pane.setEnabledAt(download_pos, Configuration.get("workflow.download", false));
		}
		else {
		    tab_pane.setEnabledAt(download_pos, Configuration.get("workflow.download", true));
		}
	    }
	    if (enrich_pos != -1) {
		tab_pane.setEnabledAt(enrich_pos, ready && Configuration.get("workflow.enrich", false));
	    }
	    if (design_pos != -1) {
		tab_pane.setEnabledAt(design_pos, ready && Configuration.get("workflow.design", false) && Configuration.getMode() > Configuration.ASSISTANT_MODE);
	    }
	    if (create_pos != -1) {
		tab_pane.setEnabledAt(create_pos, ready && Configuration.get("workflow.create", false));
	    }
	    if (format_pos != -1) {
		tab_pane.setEnabledAt(format_pos, ready && Configuration.get("workflow.format", false) && Configuration.getMode() > Configuration.ASSISTANT_MODE);
	    }
	    if (files_pos != -1) {
		tab_pane.setEnabledAt(files_pos, ready && Configuration.get("workflow.files", false) && Configuration.getMode() > Configuration.LIBRARIAN_MODE);
	    }
	}

	public void setReady(boolean ready) {
	    this.ready = ready;
	}
    }

    private class WorkflowUpdater
	implements Runnable {
	private boolean state;
	private String raw;
	public WorkflowUpdater(String raw, boolean state) {
	    this.raw = raw;
	    this.state = state;
	}
	public void run() {
	    setTabEnabled(raw, state);
	}
    }
}
