/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * <BR><BR>
 *
 * Author: John Thompson, Greenstone Digital Library, University of Waikato
 *
 * <BR><BR>
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * <BR><BR>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * <BR><BR>
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * <BR><BR>
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.gui;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import org.greenstone.gatherer.Configuration;
import org.greenstone.gatherer.DebugStream;
import org.greenstone.gatherer.Dictionary;
import org.greenstone.gatherer.Gatherer;
import org.greenstone.gatherer.cdm.Argument;
import org.greenstone.gatherer.cdm.ArgumentControl;
import org.greenstone.gatherer.collection.ScriptOptions;
import org.greenstone.gatherer.collection.Collection;
import org.greenstone.gatherer.collection.CollectionManager;
import org.greenstone.gatherer.util.AppendLineOnlyFileDocument;
import org.greenstone.gatherer.util.AppendLineOnlyFileDocumentOwner;
import org.greenstone.gatherer.util.StaticStrings;

/** This class serves as the data holder for all subclasses of option panes, such as Import options or All options. It also contains methods for creating each of the option lines as they would appear in the subpane. Futhermore it has a method for considering all the arguments and generating a <strong>String[]</strong> to allow you to pass them to the <strong>GShell</strong>.
 * @author John Thompson, Greenstone Digital Library, University of Waikato
 * @version 2.2
 */
public class OptionsPane
    extends JPanel 
    implements AppendLineOnlyFileDocumentOwner, MouseListener {

    static final public char SUCCESSFUL = 's';
    static final public char UNSUCCESSFUL = 'u';
    static final public char CANCELLED = 'c';
    static final public char UNKNOWN = 'x';
    static final public char SCHEDULED = 'd'; 

    static private int BUILD = 0;
    static private int IMPORT = 1;
    static private int SCHEDULE = 2; 
    static private int MINIMUM_ROWS = 15;
    
    /** All process messages are written to this log text area. */
    public JTextArea log_textarea = null;

    private ArrayList current_controls;

    /** The <strong>ScriptOptions</strong> data object contains all the option settings we wish to persist between Gatherer sessions (and thus is stored in <strong>Collection</strong>). */
    private ScriptOptions build_options = null;
    private ScriptOptions import_options = null;
    private ScriptOptions schedule_options = null;

    private FileEntry file_entry = null;

    /** the log pane - we only create it once now, not each time */
    private JPanel log_pane = null;
    /** the list of previous log messages */
    private JList log_list = null;
    private Vector writing_documents;


    /** The default constructor creates the few session length options, but either retrieves the rest from the current collection, or creates a default set of options. */
    public OptionsPane(ScriptOptions import_options, ScriptOptions build_options, ScriptOptions schedule_options) {
	this.build_options = build_options;
	this.import_options = import_options;
	this.schedule_options = schedule_options; 
	this.current_controls = new ArrayList();
	this.writing_documents = new Vector();
        this.setComponentOrientation(Dictionary.getOrientation());
        
	// Have to do this here, not in display, as the message log view may not have been displayed yet.
	log_textarea = new JTextArea();
        log_textarea.setComponentOrientation(Dictionary.getOrientation());
	log_textarea.setEditable(false);

	// Need to set the name for the expert mode's log_textarea
	TestingPreparation.setIndividualSubcomponentNames(this, log_textarea);
    }

    /** This method creates the panel with all the build only options on it.
     * @param pane a JPanel which already has previous arguments available on it, to allow for lower moes to concatenate all of the arguments together
     * @return a JPanel which can be used to display all the build only options
     * @see org.greenstone.gatherer.Configuration#EXPERT_MODE
     * @see org.greenstone.gatherer.Configuration#getColor
     * @see org.greenstone.gatherer.Configuration#getMode
     * @see org.greenstone.gatherer.Gatherer#config
     * @see org.greenstone.gatherer.cdm.Argument
     * @see org.greenstone.gatherer.collection.BuildOptions#getBuildArgument
     * @see org.greenstone.gatherer.collection.BuildOptions#getBuildArgumentCount
     * @see org.greenstone.gatherer.collection.BuildOptions#getBuildValue
     * @see org.greenstone.gatherer.collection.BuildOptions#getBuildValueEnabled
     * @see org.greenstone.gatherer.gui.OptionsPane.MyArgumentControl
     */
    public JPanel buildBuild(JPanel pane) {
      // Reset the arguments
      if(pane == null) {
        current_controls.clear();
      }
      ArrayList build_arguments = new ArrayList();
      int current_mode = Configuration.getMode();
      int total_build_argument_count = build_options.getArgumentCount();
      
      for(int i = 0; i < total_build_argument_count; i++) {
        // Retrieve the argument so we know how to format the control.
        Argument argument = build_options.getArgument(i);
        
        if(!argument.isHiddenGLI() && argument.getModeLevel() <= current_mode) {
          // Now attempt to retrieve any existing value for this argument.
          boolean enabled = build_options.getValueEnabled(argument.getName());
          String value = build_options.getValue(argument.getName());
          MyArgumentControl argument_control = new MyArgumentControl(BUILD, argument, enabled, value);
          build_arguments.add(argument_control);
        }
      }
      current_controls.addAll(build_arguments);
      
      // Now that we know how many arguments there are we can build the pane to view them on. Modes lower than EXPERT can provide a previous pane on which to add the arguments.
      if(pane == null || current_mode >= Configuration.EXPERT_MODE) {
        pane = new JPanel();
        pane.setComponentOrientation(Dictionary.getOrientation());
        pane.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
        pane.setBackground(Configuration.getColor("coloring.collection_tree_background", false));
        int argument_count = build_arguments.size();
        // If in any of the higher detail modes, and assuming we don't want super phat argument controls, we better ensure there is a minimum number of lines in the grid layout
        if(current_mode >= Configuration.EXPERT_MODE) {
          if(argument_count < MINIMUM_ROWS) {
            argument_count = MINIMUM_ROWS;
          }
          pane.setLayout(new GridLayout(argument_count, 1, 5, 5));
        }
        // Otherwise we're just going to throw them on one after another and chuck it in a scroll pane anyway
        else {
          // use  GridLayout with 0 rows == as many rows as needed. Unfortunately, rows will
          // grow fat if space too large, but we don't know how many rows we'll need yet.
          pane.setLayout(new GridLayout(0, 1, 5, 5));
        }
      }
      
      for(int j = 0; j < build_arguments.size(); j++) {
        pane.add((JComponent)build_arguments.get(j));
      }
      pane.addMouseListener(this);
      build_arguments = null;
      return pane;
    }

    /**Wendy's attempt at writing a buildSchedule Jpanel. */
    public JPanel buildSchedule(JPanel pane) {
	//reset the arguments
	if(pane ==  null) {
	    current_controls.clear(); 
	}

	ArrayList schedule_arguments = new ArrayList(); 
	int current_mode = Configuration.getMode(); 

	int total_schedule_argument_count = schedule_options.getArgumentCount();
      
      for(int i = 0; i < total_schedule_argument_count; i++) {
        // Retrieve the argument so we know how to format the control.
        Argument argument = schedule_options.getArgument(i);
        
       if(!argument.isHiddenGLI() && argument.getModeLevel() <= current_mode) {
          // Now attempt to retrieve any existing value for this argument.
          boolean enabled = schedule_options.getValueEnabled(argument.getName());
          String value = schedule_options.getValue(argument.getName());
          MyArgumentControl argument_control = new MyArgumentControl(SCHEDULE, argument, enabled, value);
          schedule_arguments.add(argument_control);
        }
      }
      current_controls.addAll(schedule_arguments);
      
      // Now that we know how many arguments there are we can build the pane to view them on. Modes lower than EXPERT can provide a previous pane on which to add the arguments.
      if(pane == null || current_mode >= Configuration.EXPERT_MODE) {
        pane = new JPanel();
        pane.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
        pane.setBackground(Configuration.getColor("coloring.collection_tree_background", false));
        int argument_count = schedule_arguments.size();
        // If in any of the higher detail modes, and assuming we don't want super phat argument controls, we better ensure there is a minimum number of lines in the grid layout
        if(current_mode >= Configuration.EXPERT_MODE) {
          if(argument_count < MINIMUM_ROWS) {
            argument_count = MINIMUM_ROWS;
          }
          pane.setLayout(new GridLayout(argument_count, 1, 5, 5));
        }
        // Otherwise we're just going to throw them on one after another and chuck it in a scroll pane anyway
        else {
          // use  GridLayout with 0 rows == as many rows as needed. Unfortunately, rows will
          // grow fat if space too large, but we don't know how many rows we'll need yet.
          pane.setLayout(new GridLayout(0, 1, 5, 5));
        }
      }
      
      for(int j = 0; j < schedule_arguments.size(); j++) {
        pane.add((JComponent)schedule_arguments.get(j));
      }
      pane.addMouseListener(this);
      schedule_arguments = null;
      return pane;

    }

    /** This method creates the panel with all the import only options on it.
     * @param pane a JPanel which already has previous arguments available on it, to allow for lower moes to concatenate all of the arguments together
     * @return a JPanel which can be used to display all the build only options
     * @see org.greenstone.gatherer.Configuration#EXPERT_MODE
     * @see org.greenstone.gatherer.Configuration#getColor
     * @see org.greenstone.gatherer.Configuration#getMode
     * @see org.greenstone.gatherer.Gatherer#config
     * @see org.greenstone.gatherer.cdm.Argument
     * @see org.greenstone.gatherer.collection.BuildOptions#getImportArgument
     * @see org.greenstone.gatherer.collection.BuildOptions#getImportArgumentCount
     * @see org.greenstone.gatherer.collection.BuildOptions#getImportValue
     * @see org.greenstone.gatherer.collection.BuildOptions#getImportValueEnabled
     * @see org.greenstone.gatherer.gui.OptionsPane.ArgumentControl
     */
    public JPanel buildImport(JPanel pane) {
	// Reset the arguments
	if(pane == null) {
	    current_controls.clear();
	}
	ArrayList import_arguments = new ArrayList();
	int current_mode = Configuration.getMode();
	int total_import_argument_count = import_options.getArgumentCount();       
	for(int i = 0; i < total_import_argument_count; i++) {
	    // Retrieve the argument so we know how to format the control.
	    Argument argument = import_options.getArgument(i);
	    if(!argument.isHiddenGLI() && argument.getModeLevel() <= current_mode) {
		// Now attempt to retrieve any existing value for this argument.
		boolean enabled = import_options.getValueEnabled(argument.getName());
		String value = import_options.getValue(argument.getName());
		MyArgumentControl argument_control = new MyArgumentControl(IMPORT, argument, enabled, value);
		import_arguments.add(argument_control);
	    }
	}
	current_controls.addAll(import_arguments);
	// Now that we know how many arguments there are we can build the pane to view them on. Modes lower than EXPERT can provide a previous pane on which to add the arguments.
	if(pane == null || current_mode >= Configuration.EXPERT_MODE) {
	    pane = new JPanel();
            pane.setComponentOrientation(Dictionary.getOrientation());
	    pane.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
	    pane.setBackground(Configuration.getColor("coloring.collection_tree_background", false));
	    int argument_count = import_arguments.size();	    
	    // If in any of the higher detail modes, and assuming we don't want super phat argument controls, we better ensure there is a minimum number of lines in the grid layout
	    if(current_mode >= Configuration.EXPERT_MODE) {
		if(argument_count < MINIMUM_ROWS) {
		    argument_count = MINIMUM_ROWS;
		}
		pane.setLayout(new GridLayout(argument_count, 1, 5, 5));
	    }
	    // Otherwise we're just going to throw them on one after another and chuck it in a scroll pane anyway
	    else {
		// use  GridLayout with 0 rows == as many rows as needed. Unfortunately, rows will 
		// grow fat if space too large, but we don't know how many rows we'll need yet.
		pane.setLayout(new GridLayout(0, 1, 5, 5));
	    }
	} 

	for(int j = 0; j < import_arguments.size(); j++) {
	    pane.add((JComponent)import_arguments.get(j));
	}
	pane.addMouseListener(this);
	import_arguments = null;
	return pane;
    }

    /** This method is used to build a panel based on the message log, which is nothing like any of the other panels.
     * @return A <strong>JPanel</strong> containing a scrollable text area which represents the shell process message log.
     */
    public JPanel buildLog() {
	// we now save the log pane
	if (log_pane == null) {
	    log_pane = new JPanel(new BorderLayout());
            log_pane.setComponentOrientation(Dictionary.getOrientation());
	    // Build a list of the log files available, ordering by last modified. Log files are like build_log.date.txt
	    DefaultListModel contents = new DefaultListModel();
	    File log_directory = new File(CollectionManager.getLoadedCollectionLogDirectoryPath());
	    File children[] = log_directory.listFiles();
	    for(int i = 0; children != null && i < children.length; i++) {
		String filename = children[i].getName();
		if(filename.startsWith("build_log.") && filename.endsWith(".txt") ) {
		    String datestamp = filename.substring(filename.indexOf(".") + 1, filename.lastIndexOf(".")).toLowerCase();
		    if(datestamp.indexOf("s") == -1 && datestamp.indexOf("u") == -1 && datestamp.indexOf("c") == -1 && datestamp.indexOf("x") == -1) {
			FileEntry entry = new FileEntry(children[i].getName(), children[i].getAbsolutePath());
			// We are about to insert it. But where.
			boolean found = false;
			for(int j = 0; !found && j < contents.size(); j++) {
			    FileEntry sibling = (FileEntry) contents.getElementAt(j);
			    int order = entry.compareTo(sibling);
			    if(order > 0) {
				contents.insertElementAt(entry, j);
				found = true;
			    }
			}
			if(!found) {
			    contents.addElement(entry);
			}
		    }
		}
	    }

	    log_list = new JList(contents);
            log_list.setComponentOrientation(Dictionary.getOrientation());
	    log_list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
	    log_list.setLayoutOrientation(JList.VERTICAL);
	    log_list.setPreferredSize(new Dimension(600, 100));
	    log_list.setVisibleRowCount(3);
	    log_list.addListSelectionListener(new LogListListener());

	    JLabel log_history_label = new JLabel(Dictionary.get("OptionsPane.LogHistory"));
	    log_history_label.setComponentOrientation(Dictionary.getOrientation());
            JPanel log_history_pane = new JPanel();
            log_history_pane.setComponentOrientation(Dictionary.getOrientation());
	    log_history_pane.setPreferredSize(new Dimension(600, 100));
	    log_history_pane.setLayout(new BorderLayout());
	    log_history_pane.add(log_history_label, BorderLayout.NORTH);
            JScrollPane scrol_tmp=new JScrollPane(log_list);
            scrol_tmp.setComponentOrientation(Dictionary.getOrientation());
	    log_history_pane.add(scrol_tmp, BorderLayout.CENTER);
	    
            scrol_tmp=new JScrollPane(log_textarea);
            scrol_tmp.setComponentOrientation(Dictionary.getOrientation());
	    log_pane.add(scrol_tmp, BorderLayout.CENTER);
	    log_pane.add(log_history_pane, BorderLayout.SOUTH);
	}
	return log_pane;
    }

    public AppendLineOnlyFileDocument createNewLogDocument() {
	long time = System.currentTimeMillis();
	StringBuffer name = new StringBuffer();
	name.append("build_log.");
	name.append(time);
	name.append(".txt");
	// just in case there is no log directory
	File file = new File(CollectionManager.getLoadedCollectionLogDirectoryPath() + name.toString());
	File parent_file = file.getParentFile();
	parent_file.mkdirs();
	parent_file = null;
	// create the file entry and add it to the list at pos 0 - it will always be the newest one created
	file_entry = new FileEntry(name.toString(), file.getAbsolutePath());
	((DefaultListModel)log_list.getModel()).add(0, file_entry);
	log_list.setSelectedIndex(0);
	// Finally retrieve and return the document associated with this file entry
	return file_entry.getDocument();
    }
	
	/** ensure the current build_log file has finished transferring from memory to disk, and that the random access
	*  file is properly closed when a collection is closed.  Else we can't delete the just-closed collection since the 
	* build_log file resource is still kept open. */
	public void closeCurrentLogDocument() {
		if(file_entry == null) {
			return;
		}
		AppendLineOnlyFileDocument buildDoc = file_entry.getDocument();
		if(buildDoc != null) {
			buildDoc.close();
		}		
	}


    /** Attempts to discover the latest document count.
     * @return An <strong>int</strong> detailing the number of documents in this collection.
     */
    public int getDocumentCount() {
	if(Gatherer.c_man.ready()) {
	    int count = Gatherer.c_man.getCollection().getDocumentCount();
	    if(count != 0) {
		return count;
	    }
	}
	return 1;
    }

    /** Called by our magic log documents after they have finished writing themselves to file, whereapon it is no longer necessary to hold a reference to them. */
    public void remove(AppendLineOnlyFileDocument document) {
	writing_documents.remove(document);
    }

    public void resetFileEntry() {
	if(file_entry != null) {
	    file_entry.reset();
	}
    }

    /** Given a panel containing ArgumentControls, update the values associated with them. */
    public void update(JPanel panel) {
	if(panel == log_pane) {
	    return;
	}

	for(int i = 0; i < panel.getComponentCount(); i++) {
	    Component component = panel.getComponent(i);
	    if(component instanceof MyArgumentControl) {
		((MyArgumentControl)component).update();
	    }
	}
    }

    /** Implementation side-effect
     * @param e a MouseEvent
     */
    public void mouseClicked(MouseEvent e) {}

    /** Implementation side-effect
     * @param e a MouseEvent
     */
    public void mouseEntered(MouseEvent e) {}

    /** Implemented to ensure that, by the time the mouse pointer leaves the current build options screen, ang changes to the value the JSpinners have been commited
     * @param e a MouseEvent
     */
    public void mouseExited(MouseEvent e) {
	// Loop through the controls, and if the current control is a JSpinner, commit its current editing
	for(int i = 0; i < current_controls.size(); i++) {
	    MyArgumentControl control = (MyArgumentControl) current_controls.get(i);
	    JComponent value_control = control.getValueControl();
	    if(value_control instanceof JSpinner) {
		try {
		    ((JSpinner)value_control).commitEdit();
		}
		catch(Exception exception) {
		    DebugStream.println("Exception in OptionsPane.mouseExited() - unexpected");
		    DebugStream.printStackTrace(exception);
		}
	    }
	    value_control = null;
	    control = null;
	}
    }

    /** Implementation side-effect
     * @param e a MouseEvent
     */
    public void mousePressed(MouseEvent e) {}

    /** Implementation side-effect
     * @param e a MouseEvent
     */
    public void mouseReleased(MouseEvent e) {}

    private class MyArgumentControl
	extends ArgumentControl {
	private int type;

	public MyArgumentControl(int type, Argument argument, boolean enable, String value) {
	    super(argument, enable, value);
	    this.type = type;
	}

	/** Update the values stored in the collection so as to remember the current state of this argument. */
	public void update() {
	    String name = getArgumentName();
	    boolean enable = isEnabled();
	    String value = getValue();
	    // If this argument was a flag, but is now disabled, remove from the build options altogether
	    if(!enable && value == null) {
		if(type == BUILD) {
		    build_options.removeValue(name);
		}
		else if(type == SCHEDULE) {
		    schedule_options.removeValue(name); 
		}
		else {
		    import_options.removeValue(name);
		}
	    }
	    // Otherwise update the argument value
	    else {
		if(type == BUILD) {
		    build_options.setValue(name, enable, value);
		}
		else if(type == SCHEDULE) {
		    schedule_options.setValue(name, enable, value); 
		}
		else {
		    import_options.setValue(name, enable, value);
		}
	    }
	}
    }

    /** Holds a File which has a particular naming convention build_log.date.txt also keeps a Date corresponding to the date in its name*/
    private class FileEntry {

       private AppendLineOnlyFileDocument current_document;
       private Date date;
       private long last_modified;
       private String display;
       private String filename;
       private String filepath;
       
       public FileEntry(String filename, String filepath) {
	  this.date = null;
	  this.display = null;
	  this.filename = filename;
	  this.filepath = filepath;
	  this.last_modified = 0L;
       }

	/** returns 0 if the dates are the same, -ve number if the current FileEntry is earlier than the fe FileEntry ...*/
	public int compareTo(FileEntry file_entry) {
	    Date our_date = getDate();
	    Date other_date = file_entry.getDate();
	    return our_date.compareTo(other_date);
	}

	public Date getDate() {
	    if(date == null) {
		// Need to exclude first '.'
		int first_index = filename.indexOf(".") + 1;
		// Need to exclude the last '.'
		int last_index = filename.lastIndexOf(".");
		if(first_index > 0 && last_index > 0 && first_index < last_index) {
		    String date_string = filename.substring(first_index, last_index);
		    date = new Date(Long.parseLong(date_string));
		}
		else {
		    date = new Date(); // Current date
		}
	    }
	    return date;
	}

       public AppendLineOnlyFileDocument getDocument() {
	  if(current_document == null) {
	     current_document = new AppendLineOnlyFileDocument(filepath);
	  }
	  return current_document;
	}

	public void reset() {
	    display = null;
	}

	/** we only want the date out of the file name, not the whole path */
	public String toString() {
	    File file = new File(filename);
	    if(display == null) {
		last_modified = file.lastModified();
		StringBuffer d = new StringBuffer();
		Date date = getDate();
		d.append(date.toString());
		char success = UNKNOWN;
		File the_file = new File(filepath);
		if(the_file.exists()) {
		   try {
		      FileInputStream in = new FileInputStream(the_file);
		      success = (char) in.read();
		      in.close();
		      in = null;
		   }
		   catch(Exception error) {
		      ///ystem.err.println("Log '" + filepath + "' not found!");
		      ///atherer.printStackTrace(error);
		   }
		}
		the_file = null;
		switch (success) {
		case SUCCESSFUL:
		    d.append(Dictionary.get("OptionsPane.Successful"));
		    break;
		case UNSUCCESSFUL:
		    d.append(Dictionary.get("OptionsPane.Unsuccessful"));
		    break;
		case CANCELLED:
		    d.append(Dictionary.get("OptionsPane.Cancelled"));
		    break;
		case SCHEDULED:
		    d.append(Dictionary.get("OptionsPane.Scheduled"));
		    break;
		default:
		    d.append(Dictionary.get("OptionsPane.Unknown"));
		}
		display = d.toString();
	    }
	    return display;
	}
    }

    /** a ListSelectionListener that triggers the load of a newly selected log */
    private class LogListListener implements ListSelectionListener {

	public void valueChanged(ListSelectionEvent e) {
	    if (!e.getValueIsAdjusting()) { // we get two events for one change in list selection - use the false one ( the second one)
		///ystem.err.println("Log change detected.");
		JList source  = (JList)e.getSource();
		file_entry = (FileEntry) source.getSelectedValue();
		// First we determine if the old log has been completely written to file
		Document document = log_textarea.getDocument();
		///ystem.err.println(" * current document: " + document);
		///ystem.err.println(" * new document:     " + file_entry.getDocument());
		// If we are dealing with the same document don't do anything.
		if(document != file_entry.getDocument()) {
		    if(document instanceof AppendLineOnlyFileDocument) {
			AppendLineOnlyFileDocument append_line_only_file_document = (AppendLineOnlyFileDocument) document;
			if(append_line_only_file_document.isStillWriting()) {
			    ///ystem.err.println("Current log is still active... finishing.");
			    writing_documents.add(append_line_only_file_document); // We have to maintain a reference until they are all done.
			    append_line_only_file_document.setOwner(OptionsPane.this);
			    append_line_only_file_document.setExit();
			}
			else {
			    ///ystem.err.println("Current log is complete. Nothing to do.");
			}
		    }
		    // Load the new log
		    log_textarea.setDocument(file_entry.getDocument());
		}
	    }
	}
    }

}
