/**********************************************************************
 *
 * cgiargs.h -- 
 * Copyright (C) 1999  The New Zealand Digital Library Project
 *
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/


#ifndef CGIARGS_H
#define CGIARGS_H

#include "gsdlconf.h"
#include "text_t.h"

#if defined(GSDL_USE_OBJECTSPACE)
#  include <ospace\std\iostream>
#elif defined(GSDL_USE_IOS_H)
#  include <iostream.h>
#else
#  include <iostream>
#endif

#if defined(GSDL_USE_OBJECTSPACE)
#  include <ospace\std\map>
#elif defined(GSDL_USE_STL_H)
#  include <map.h>
#else
#  include <map>
#endif


// response_t is used to define what type of cgi header 
// should be produced
// fullcontent same as content but the action should put content-type as last header.
enum response_t {location, content, fullcontent, undecided_location};

struct fileupload_t {
  void clear();
  fileupload_t() {clear();}
  
  text_t name;
  text_t type;
  int size;
  text_t tmp_name;
};

typedef map<text_t, fileupload_t, lttext_t> fileupload_tmap;

struct cgiarg_t {
  void clear ();
  cgiarg_t () {clear();}
  cgiarg_t &operator=(const cgiarg_t &x);
  
  enum source_t {default_arg, compressed_arg, cgi_arg, program_arg};
  
  text_t value;
  source_t source;
  fileupload_t fileupload;
};

ostream &operator<<(ostream &outs, const fileupload_t &fu);

bool operator==(const cgiarg_t &x, const cgiarg_t &y);
bool operator<(const cgiarg_t &x, const cgiarg_t &y);

typedef map<text_t, cgiarg_t, lttext_t> cgiarg_tmap;
typedef map<text_t, text_tmap, lttext_t> confcgiarg_tmap;

// cgiargsclass is used to store a particular set
// of cgi arguments.
class cgiargsclass {
protected:
  cgiarg_tmap args;

public:
  //type support for cgiarg_tmap
  typedef cgiarg_tmap::iterator iterator;
  typedef cgiarg_tmap::const_iterator const_iterator;
  typedef cgiarg_tmap::reference reference;
  typedef cgiarg_tmap::const_reference const_reference;
  typedef cgiarg_tmap::size_type size_type;
  typedef cgiarg_tmap::difference_type difference_type;
  typedef cgiarg_tmap::const_reverse_iterator const_reverse_iterator;
  typedef cgiarg_tmap::reverse_iterator reverse_iterator;
  
  // constructors
  cgiargsclass ();

  // basic container support
  iterator begin () {return args.begin();}
  const_iterator begin () const {return args.begin();}
  iterator end () {return args.end();}
  const_iterator end () const {return args.end();}

  void erase(iterator pos) {args.erase(pos);}
  void erase(iterator first, iterator last) {args.erase(first, last);}
  cgiargsclass &operator=(const cgiargsclass &x) {args=x.args;return *this;}

  bool empty () const {return args.empty();}
  size_type size() const {return args.size();}


  // added functionality
  void clear () {args.erase(args.begin(),args.end());}

  // setdefaultarg, setdefaultintarg, and setdefaultcarg will 
  // only set the argument if it is not already set
  // getargs returns NULL if there isn't an entry with
  // 'key' already defined, getintarg returns 0 if there wasn't an
  // entry with 'key' defined and operator[] returns "" if
  // 'key' wasn't already defined (and sets 'key'="").
  void setarg (const text_t &key, const text_t &value,
	       cgiarg_t::source_t source=cgiarg_t::program_arg);
  void setdefaultarg (const text_t &key, const text_t &value,
		      cgiarg_t::source_t source=cgiarg_t::default_arg);
  void setintarg (const text_t &key, int value,
		  cgiarg_t::source_t source=cgiarg_t::program_arg);
  void setdefaultintarg (const text_t &key, int value,
			 cgiarg_t::source_t source=cgiarg_t::default_arg);
  void setcarg (const text_t &key, unsigned short c,
		cgiarg_t::source_t source=cgiarg_t::program_arg);
  void setdefaultcarg (const text_t &key, unsigned short c,
		       cgiarg_t::source_t source=cgiarg_t::default_arg);
  void setargfile (const text_t &key, const fileupload_t &fileupload);
  text_t *getarg (const text_t &key);
  int getintarg (const text_t &key);
  fileupload_t *getargfile (const text_t &key);
  text_t &operator[] (const text_t &key) {return args[key].value;}
  cgiarg_t &lookupcgiarg (const text_t &key) {return args[key];}
};

// stream operators to print cgi arguments for debugging purposes
ostream &operator<<(ostream &outs, const cgiargsclass &args);



// cgiarginfo holds information relating to a cgi argument
class cgiarginfo {
public:
  cgiarginfo ();

  // shortname is the name used in cgi arguments
  text_t shortname;

  // longname is the name used when giving information 
  // about the argument and to spot name duplication
  text_t longname;

  // multiplechar should be set to true if the value can be
  // more than one character long
  bool multiplechar;

  // multiplevalue should be set to true the argument may have 
  // multiple values (as is the case when multiple checkboxes
  // use the same name)
  bool multiplevalue;

  // fileupload should be set if the argument is to be used to upload files
  bool fileupload;

  // defaultstatus_t indicates how good the default is when different
  // defaults are given for one argument. "none" means there is no default
  // (a default might be given elsewhere). "weak" is a basic guess that
  // doesn't hold much weight, "good" is a better guess, "config" was a default
  // given in a configuration file, and "imperative" means it must not
  // be overriden at any costs (unless there is another "imperative"...)
  enum defaultstatus_t {none=0, weak=1, good=2, config=3, imperative=4};

  defaultstatus_t defaultstatus;
  text_t argdefault;   // a default value

  // savedarginfo_t indicates whether the argument should be saved
  // between pages (e.g. using the compressed argument) or not. The
  // value of an argument can change from "can" to "mustnot" and 
  // "can" to "must" but not "mustnot" to "must" or "must" to "mustnot".
  enum savedarginfo_t {mustnot, can, must};

  savedarginfo_t savedarginfo;
};

bool operator==(const cgiarginfo &x, const cgiarginfo &y);
bool operator<(const cgiarginfo &x, const cgiarginfo &y);

typedef map<text_t, cgiarginfo, lttext_t> argsinfomap;

// contains a list of cgi argument information
class cgiargsinfoclass {
protected:
  argsinfomap argsinfo;

public:
  // type support for arginfomap
  typedef argsinfomap::iterator iterator;
  typedef argsinfomap::const_iterator const_iterator;
  typedef argsinfomap::reference reference;
  typedef argsinfomap::const_reference const_reference;
  typedef argsinfomap::size_type size_type;
  typedef argsinfomap::difference_type difference_type;
  typedef argsinfomap::const_reverse_iterator const_reverse_iterator;
  typedef argsinfomap::reverse_iterator reverse_iterator;
  
  // constructors
  cgiargsinfoclass ();

  // basic container support
  iterator begin () {return argsinfo.begin();}
  const_iterator begin () const {return argsinfo.begin();}
  iterator end () {return argsinfo.end();}
  const_iterator end () const {return argsinfo.end();}

  void erase(iterator pos) {argsinfo.erase(pos);}
  void erase(iterator first, iterator last) {argsinfo.erase(first, last);}
  cgiargsinfoclass &operator=(const cgiargsinfoclass &x) {argsinfo=x.argsinfo;return *this;}

  bool empty () const {return argsinfo.empty();}
  size_type size() const {return argsinfo.size();}


  // added functionality
  void clear () {argsinfo.erase(argsinfo.begin(),argsinfo.end());}

  // addarginfo will combine the information with the present
  // information. If name clashes were detected then the information
  // will be written to logout and addarginfo will return false. No
  // processing with the arguments should be done if this happens
  // as the results will be meaningless.
  bool addarginfo (ostream *logout, cgiarginfo &info);
  bool addarginfo (ostream *logout, cgiargsinfoclass &info);

  // addarginfo will override args info with info loaded from config files
  // if the args do not exists will be created
  bool addarginfo (ostream *logout, const text_t& argshortname, const text_tmap& mapinfo);
  bool addarginfo (ostream *logout, confcgiarg_tmap& info);

  cgiarginfo *getarginfo (const text_t &key);
  const cgiarginfo *getarginfo (const text_t &key) const;
  cgiarginfo &operator[] (const text_t &key) {return argsinfo[key];}
};

ostream &operator<<(ostream &outs, const cgiargsinfoclass &argsinfo);

#endif
