/* ------------------------------------------------------------------- */
/* D2Mmain   : standalone program for DC -> MARC conversion            */
/*                                                                     */
/* Syntax    : d2main [File] [Options]                                 */
/*                                                                     */
/* File      : This may be a local filename, or a URL (which is        */
/*             recognized by the "http://" prefix). When a URL,        */
/*             the remote file will be fetched using the builtin       */
/*             HTTP "client".                                          */
/*                                                                     */
/* Options   : -l                       : Format = line format         */
/*                                        (Default is ISO 2709)        */
/* Options   : -f [filename]            : Outputfile                   */
/*                                        (Default is stdout)          */
/* Options   : -m [marcformat]          : Marcformat (upper case)      */
/*                                        (Default is NORMARC)         */
/* Options   : -t [filename]            : Tracefile                    */
/*                                        (Default is no trace)        */
/*                                                                     */
/* Return    : 0 : OK                                                  */
/*             1 : Syntax error (no parms != 2)                        */
/*             2 : Illegal MARC format                                 */
/*             3 : Cannot open output file                             */
/*             8 : Error in MARC conversion                            */
/*             9 : URL not found                                       */
/*                                                                     */
/* Author    : Ole Husby                                               */
/* Updated   : 1998-09-30                                              */
/* ------------------------------------------------------------------- */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <unistd.h>
#include "d2m.h"





/* ------------------------------------------------------------------- */
/* main() program                                                      */
/* ------------------------------------------------------------------- */

int main (int argc, char **argv)
{
  int rc, i, f2709, tracing, argtype, mformat, marcfile;
  int linel, llen, slen;

  FILE *df, *of;
  char dfilename[64], tfilename[64], ofilename[64], line[50000];
  char *pq, *p, *q, pre[16], *l, *u;
  char marcformat[64], url[1000];
  char buffer[50000];

  char recstatus = 'n';
  char rectype   = 'm';
  char biblevel  = ' ';

  enum options {OPT_NONE, OPT_FILE, OPT_TRAC, OPT_MARC};


/* Initialize */

  of = stdout;
  *url = *ofilename = 0;
  strcpy(marcformat, "NORMARC");

  tracing = FALSE;
  *tfilename = 0;
  f2709 = TRUE;
  marcfile = 0;

  *line = 0;

  linel = D2M_LINELEN;
  if (!linel)
    linel = 80;

  if (linel > 200)
    linel = 200;

  llen = linel - 9;
  slen = linel - 30;
  if (slen < 10)
    slen = 10;


/* Read command line arguments */

  if (argc < 2)
  { 
    printf(" Syntax    : d2main [File] [Options]\n\n");
    printf(" File      : This may be a local filename, or a URL (which is\n");
    printf("             recognized by the \"http://\" prefix). When a URL,\n");
    printf("             the remote file will be fetched using the builtin\n");
    printf("             HTTP \"client\".\n\n");
    printf(" Options   : -l                       : Format = line format\n");
    printf("                                        (Default is ISO 2709)\n");
    printf(" Options   : -f [filename]            : Outputfile\n");
    printf("                                        (Default is stdout)\n");
    printf(" Options   : -m [marcformat]          : Marcformat (upper case)\n");
    printf("                                        (Default is NORMARC)\n");
    printf(" Options   : -t [filename]            : Tracefile\n");
    printf("                                        (Default is no trace)\n\n");
    printf(" Return    : 0 : OK\n");
    printf("             1 : Syntax error\n");
    printf("             2 : Illegal MARC format\n");
    printf("             3 : Cannot open output file\n");
    printf("             8 : Error in MARC conversion\n");
    printf("             9 : URL not found\n");
    exit(1);
  }

  strcpy(dfilename, argv[1]);


/* Get options                                                         */

  if (argc > 2)
  {
    argtype = OPT_NONE;

    for (i = 1; i < argc; i++)
    {
      if (argtype == OPT_FILE)
      {
        strcpy(ofilename, argv[i]);
        argtype = OPT_NONE;
      }
      else if (argtype == OPT_TRAC)
      {
        strcpy(tfilename, argv[i]);
        tracing = TRUE;
        argtype = OPT_NONE;
      }
      else if (argtype == OPT_MARC)
      {
        strcpy(marcformat, argv[i]);
        argtype = OPT_NONE;
      }
      else if (strcmp(argv[i], "-l") == 0)
      {
        f2709 = FALSE;
        argtype = OPT_NONE;
      }
      else if (strcmp(argv[i], "-m") == 0)
         argtype = OPT_MARC;
      else if (strcmp(argv[i], "-f") == 0)
         argtype = OPT_FILE;
      else if (strcmp(argv[i], "-t") == 0)
         argtype = OPT_TRAC;
      else
        argtype = OPT_NONE;
    }
  }

/* Verify MARC format */

  if (strncasecmp(marcformat, "US", 2) == 0)
    mformat = USMARC;
  if (strncasecmp(marcformat, "UNI", 3) == 0)
    mformat = UNIMARC;
  else if (strncasecmp(marcformat, "DAN", 3) == 0)
    mformat = DANMARC;
  else if (strncasecmp(marcformat, "FIN", 3) == 0)
    mformat = FINMARC;
  else if (strncasecmp(marcformat, "IS", 2) == 0)
    mformat = ISMARC;
  else if (strncasecmp(marcformat, "NOR", 3) == 0)
    mformat = NORMARC;
  else if (strncasecmp(marcformat, "SWE", 3) == 0)
    mformat = SWEMARC;
  else 
    exit(2);


/* Fetch URL if remote file  */

  if (strncmp(dfilename, "http://", 7) == 0)
  {
    strcpy(url, dfilename);
    sprintf(dfilename, "%s.%d", D2M_TEMPFILE, getpid());
    rc = wfetch(&url, &dfilename);
    if (!rc)
    {
      remove(dfilename);
      exit(9);
    }
    if (rc == 2)
      marcfile = 1;
  }




/* Convert into buffer (line format)  */

  if (marcfile)
    rc = M2Mconv((char *) dfilename, (char *) buffer,
                (char *) tfilename, mformat);
  else
    rc = D2Mconv((char *) dfilename, (char *) buffer,
                (char *) tfilename, (char *) url, mformat);

  if (!rc)
  {
    fclose(df);
    exit(8);
  }

  if (*url)
    remove(dfilename);


/* Prepare for output      */

   if (*ofilename)
   {
     of = fopen(ofilename, "w");
     if (!of)
       exit(3);
   }
   else
     of = stdout;


/* Write output, if necessary converted to ISO 27809   */

   if (f2709)
     fprintf(of, "%s", 
       (char *) MARC2709(buffer, 0, 6, recstatus, rectype, biblevel));

   else
   {
     l = strtok((char *) buffer, "\n");

     while (l)
     {
       write_field(l, of, linel, llen, slen);
       l = strtok(NULL, "\n");
     }
   }

   if (*ofilename)
     fclose(of);
}
