/**********************************************************************
 *
 * BlkFile.h
 * Copyright (C) 2003  UNESCO
 *
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/

//////////////////////////////////////////////////////////////////////
//  BLKFILE.H
//
//  Class for accessing a file per blocks of blksize bytes, blocks
//  are sequential from 1 to nblock.
//                                                                       
//////////////////////////////////////////////////////////////////////
 
#ifndef _BLKFILE_H_
#define _BLKFILE_H_
#include "CacheMan.h"
#include "File.h"
 
//////////////////////////
// Block I/O file class
//
 
 
 
class BlkFile : public CFileBase
{
public:
   enum { blk_size = 512 };
protected:
   unsigned     blkSize_;      // Block Size
   CacheManager cache_;
public:
   BlkFile(int bs=BlkFile::blk_size);
   BlkFile(const _TCHAR *fname, int bs=BlkFile::blk_size);
   virtual FileError Close();
   virtual ~BlkFile();
   virtual void ReadBlk(void* d, mg_s_long b);
   virtual void WriteBlk(void* d, mg_s_long b);
   virtual void ReadBlk(void* d);
   virtual void WriteBlk(void* d);

   void SetBlockSize(int blockSize) { blkSize_ = blockSize; }
   int  GetBlockSize() { return blkSize_; }
};
 
////////////////////////////////////////
 
inline BlkFile::BlkFile(int bs) :
// Makes a Blocked File object that isn't connected to a file
// Block size is bs
CFileBase(), blkSize_(bs)
{
	cache_.Create(bs, 40);
	cache_.Connect(this);
   // Nothing more to do
}
 
 
////////////////////////////////////////
 
inline BlkFile::BlkFile(const _TCHAR *fname, int bs) :
// Makes a Blocked File object connected to a file
// Block size is bs
CFileBase(), blkSize_(bs), cache_(bs, 20)
{
	CFileBase::Open(fname, FileSystem::FILE_READWRITE);
	cache_.Create(bs, 40);
	cache_.Connect(this);
   // Nothing more to do
}
 
////////////////////////////////////////
 
inline BlkFile::~BlkFile()
// Destructor  for BlkFile object
{
	cache_.Disconnect();
   // Nothing more to do
}
 
////////////////////////////////////////////


inline FileError BlkFile::Close()
{
	if (IsOpen())
	    cache_.Invalidate();
	return CFileBase::Close();
}
////////////////////////////////////////
 
inline void BlkFile::ReadBlk(void* d, mg_s_long b)
// Reads block number b into d, b is in range [1:NBLOCK]
{
	ASSERT(d!=NULL && b>0);
	//TRACE("\nBlkFile::ReadBlk -- %s Read Block %ld", file_name, b);
    //CFileBase::Fetch(d, blkSize_, (b-1)*blkSize_);
	cache_.Read((b-1)*blkSize_, d);
}
 
////////////////////////////////////////
 
inline void BlkFile::WriteBlk(void* d, mg_s_long b)
// Writes blksize bytes from d at block number b
{
	ASSERT(d!=NULL && b>0);
	//TRACE("\nBlkFile::WriteBlk -- %s Write Block %ld", file_name, b);
    //CFileBase::Store(d, blkSize_, (b-1)*blkSize_);
	cache_.Write((b-1)*blkSize_, d);
}
 
 
////////////////////////////////////////
 
inline void BlkFile::ReadBlk(void* d)
// Reads block at current address into d
{
	ASSERT(d!=NULL);
    //CFileBase::Fetch(d, blkSize_);
	mg_s_long addr = CFileBase::FilePosition();
	cache_.Read(addr, d);
}
 
////////////////////////////////////////
 
inline void BlkFile::WriteBlk(void* d)
// Writes blkSize_ bytes from d at current address
{
	ASSERT(d!=NULL);
    //CFileBase::Store(d, blkSize_);
	mg_s_long addr = CFileBase::FilePosition();
	cache_.Write(addr, d);
}
 
#endif
