/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Copyright (C) 2006 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.cdm;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.table.*;
import java.util.ArrayList;
import java.util.LinkedHashSet;

import org.greenstone.gatherer.Configuration;
import org.greenstone.gatherer.DebugStream;
import org.greenstone.gatherer.Dictionary;
import org.greenstone.gatherer.Gatherer;
import org.greenstone.gatherer.gui.DesignPaneHeader;
import org.greenstone.gatherer.gui.TestingPreparation;
import org.greenstone.gatherer.util.StaticStrings;


public class SearchMetadataManager
    extends CollectionMetaManager
{
    private Control controls;
 
  // the following are special and should not be translated
  public static final String METADATA_INDEX = "ex.metadata";

    //CollectionMetaManager collmeta_manager = CollectionDesignManager.collectionmeta_manager;

    public SearchMetadataManager() {
	super(CollectionDesignManager.collect_config.getDocumentElement(), StaticStrings.SEARCHMETADATA_ELEMENT, new SearchMeta("", ""));
	DebugStream.println("SearchMetaManager: " + getSize() + " metadata parsed.");
    }
    
    /** Destructor. */
    public void destroy() {
	if (controls != null) {
	    controls.destroy();
	    controls = null;
	}
    }

    public void loseFocus() {
    }

    public void gainFocus() {
    }
    
    public Control getControls() {
	if (controls == null) {
	    controls = new DisplayControl();
	}
	return controls;
    }
    
    public SearchMeta getMetadatum(String name, String type) {
	return getMetadatum(name, type, true);
    }

    public SearchMeta getMetadatum(String name, String type, boolean add_if_not_found) {
	int size = getSize();
	if (size == 0) {
	    return null;
	}
	for(int i = 0; i < size; i++) {
	    SearchMeta metadatum = (SearchMeta) getElementAt(i);
	    if(metadatum.getName().equals(name) && metadatum.getType().equals(type) && metadatum.getLanguage().equals(Configuration.getLanguage())) {
		//DebugStream.println("Found '" + metadatum + "'");
		return metadatum;
	    }
	    else {
		//DebugStream.println("No match with: " + metadatum.getName() + " [l=" + metadatum.getLanguage() + "] \"" + metadatum.getValue() + "\"");
	    }
	    metadatum = null;
	}
	if(add_if_not_found) {
	    SearchMeta result = new SearchMeta(name, type);
	    addMetadatum(result);
	    //DebugStream.println("Added new metadata: " + name);
	    return result;
	}
	else {
	    return null;
	}
    }

    /** Retrieve all of the metadata for the given feature and type */
    public ArrayList getMetadata(String name, String type) {
    	ArrayList result = new ArrayList();
    	int size = getSize(); // Refresh DOM Model
    	for(int i = 0; i < size; i++) {
    	    SearchMeta metadata = (SearchMeta) getElementAt(i);
    	    if(metadata.getName().equals(name) && metadata.getType().equals(type)) {
    		result.add(metadata);
    	    }
    	}
    	return result;
    }


    public void removeMetadata(String name, String type) {
	for(int i = getSize(); i != 0; i--) {
	    SearchMeta other = (SearchMeta) getElementAt(i - 1);
	    if(name.equals(other.getName()) && type.equals(other.getType())) {
		remove(i - 1);
	    }
	    other = null;
	}	
    }

    private class DisplayControl
	extends JPanel
	implements Control {

	private SearchMetadataTable metadata_table = null;
	
	public DisplayControl() {
	    super();
            this.setComponentOrientation(Dictionary.getOrientation());
	    JPanel header_panel = new DesignPaneHeader("CDM.GUI.SearchMetadata", "searchmetadatasettings");

	    metadata_table = new SearchMetadataTable();

	    JScrollPane scroll_panel = new JScrollPane(metadata_table);
	    scroll_panel.getViewport().setBackground(Configuration.getColor("coloring.collection_tree_background", false));
	    scroll_panel.setOpaque(true);

	    JPanel metadata_table_pane = new JPanel();
            metadata_table_pane.setComponentOrientation(Dictionary.getOrientation());
	    metadata_table_pane.setBorder(BorderFactory.createEmptyBorder(5,0,0,0));
	    metadata_table_pane.setLayout(new BorderLayout());
	    metadata_table_pane.add(scroll_panel, BorderLayout.CENTER);

	    setBorder(BorderFactory.createEmptyBorder(0,5,0,0));
	    setLayout(new BorderLayout());
	    add(header_panel, BorderLayout.NORTH);
	    add(metadata_table_pane, BorderLayout.CENTER);

	    TestingPreparation.setNamesRecursively(this);
	}

	public void loseFocus(){
	    // save the last metadata in case its still editing
	    metadata_table.stopEditing();
	}
	
	
	public void gainFocus(){
	    // refresh the model as we may have changed indexes in the meantime
	    metadata_table.refreshModel();
	}

	public void destroy() {}
    }
    
    private ArrayList getEntries() {
	LinkedHashSet entries = new LinkedHashSet();
	
	ArrayList indexes = CollectionDesignManager.index_manager.getIndexes();
	if (indexes != null) {
	    int indexes_size = indexes.size();
	    for (int i=0; i<indexes_size; i++) {
	      String index_id = ((Index)indexes.get(i)).getID();
	      if (!index_id.equals(METADATA_INDEX)) {
		SearchMetadataEntry sme = new SearchMetadataEntry(index_id, SearchMeta.TYPE_INDEX);
		entries.add(sme);
	      }
	    }
	}
	ArrayList sortfields = CollectionDesignManager.index_manager.getSortFields();
	if (sortfields != null) {
	    int sortfields_size = sortfields.size();
	    for (int i=0; i<sortfields_size; i++) {
              String sf_id = ((Index)sortfields.get(i)).getID();
              SearchMetadataEntry sme = new SearchMetadataEntry(sf_id, SearchMeta.TYPE_SORT);
		entries.add(sme);

	    }
	}
	ArrayList facets = CollectionDesignManager.index_manager.getFacets();
	if (facets != null) {
	    int facets_size = facets.size();
	    for (int i=0; i<facets_size; i++) {
              String facet_id = ((Index)facets.get(i)).getID();
              SearchMetadataEntry sme = new SearchMetadataEntry(facet_id, SearchMeta.TYPE_FACET);
		entries.add(sme);
	    }
	}

	ArrayList levels = CollectionDesignManager.index_manager.getLevels();
	if (levels != null) {
	    int levels_size = levels.size();
	    for (int i=0; i<levels_size; i++) {
		SearchMetadataEntry sme = new SearchMetadataEntry(((IndexOption)levels.get(i)).getName(), SearchMeta.TYPE_LEVEL);
		entries.add(sme);
	    }
	}

        
	ArrayList partitions = CollectionDesignManager.subcollectionindex_manager.getSubcollectionIndexes();
	if (partitions != null) {
	    int partitions_size = partitions.size();
	    for(int i=0; i<partitions_size; i++) {
		SearchMetadataEntry sme = new SearchMetadataEntry(((SubcollectionIndex)partitions.get(i)).getID(), SearchMeta.TYPE_PARTITION);
		entries.add(sme);
	    }
	}
	
	ArrayList languages = CollectionDesignManager.language_manager.getLanguages();
	if (languages != null) {
	    int languages_size = languages.size();
	    for (int i=0; i<languages_size; i++) {
		SearchMetadataEntry sme = new SearchMetadataEntry(((Language)languages.get(i)).getCode(), SearchMeta.TYPE_LANGUAGE);
		entries.add(sme);
	    }
	}
	ArrayList list_entries = new ArrayList();
	list_entries.addAll(entries);
	return list_entries;
    }
    
    
    private class SearchMetadataEntry {
		
	String id;
	String type;
	SearchMeta coll_meta = null;
	String value;

	public SearchMetadataEntry(String id, String type) {
	    if (!Gatherer.GS3) {
		type = SearchMeta.TYPE_SEARCH;
	    }
	    this.id = id;
	    this.type = type;
	    //coll_meta = collmeta_manager.getMetadatum(getMetaID(), type);
	    this.coll_meta = getMetadatum(id, type); //getMetaID(), type);
	    if (this.coll_meta != null) {
		this.value = coll_meta.getValue(CollectionMeta.GREENSTONE);
	    }
	}
	
	public String toString() {
	    return Dictionary.get("CDM.SearchMetadataManager.Type_"+type)+": "+id;
	}
	public boolean equals(Object sme) {
	    if (sme instanceof SearchMetadataEntry) {
		if (id.equals(((SearchMetadataEntry)sme).id) && type.equals(((SearchMetadataEntry)sme).type)) {
		    return true;
		}
		return false;
	    }
	    return toString().equals(sme.toString());
	
	}
	public String getMetaID() {
	  if (Gatherer.GS3) {
	    // we don't use any dots in gs3
	    return id;
	  }
	  return StaticStrings.STOP_CHARACTER+id;
	  
	}
	public String getValue() {
	    return value;
	}
	public void setValue(String val) {
	    coll_meta.setValue(val);
	    value = val;
	}
    }

    private class SearchMetadataTable
	extends JTable {

	private SearchMetadataTableModel model = null;
	
	public SearchMetadataTable() {
	    // create the model
	  this.setComponentOrientation(Dictionary.getOrientation());
	  
	    model = new SearchMetadataTableModel();
	    setModel(model);

	    setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

	    // set our own cell renderer
	    TableColumnModel column_model = getColumnModel();
	    TableColumn name_column = column_model.getColumn(0);
	    TableColumn value_column = column_model.getColumn(1);

	    SearchMetadataTableCellRenderer cell_renderer = new SearchMetadataTableCellRenderer();
	    name_column.setCellRenderer(cell_renderer);
	    value_column.setCellRenderer(cell_renderer);
	}
	
	public void refreshModel() {
	    model.refresh();
	}

	public void stopEditing() {
	    // Save the current value in the text field, then remove the editor so it doesn't get saved again
	    TableCellEditor table_cell_editor = getCellEditor();
	    if (table_cell_editor != null) {
		table_cell_editor.stopCellEditing();
	    }
	}
    }
    
    private class SearchMetadataTableModel
	extends AbstractTableModel {
	
	// the list of items in the table
	private ArrayList search_metadata_entries = null;
	
	final private String[] COLUMN_NAMES = {Dictionary.get("CDM.SearchMetadataManager.Component"), Dictionary.get("CDM.SearchMetadataManager.Component_Name")};
	
	public SearchMetadataTableModel() {
	    refresh(); 
	}
	
	/** Returns the number of columns in this table. */
	public int getColumnCount() {
	    
	    return COLUMN_NAMES.length;
	}
	
	/** Retrieves the name of the specified column. */
	public String getColumnName(int col) {
	    
	    return COLUMN_NAMES[col];
	}
	
	/** Returns the number of rows in this table. */
	public int getRowCount() {
	    
	    return search_metadata_entries.size();
	}
	
	/** Returns the cell value at a given row and column as an Object. */
	public Object getValueAt(int row, int col) {
	    // Check values are reasonable
	    if (row < 0 || row >= search_metadata_entries.size() || col < 0 || col >= COLUMN_NAMES.length) {
		return null;
	    }
	    
	    SearchMetadataEntry sme = (SearchMetadataEntry) search_metadata_entries.get(row);
	    if (col == 0) {
		return sme.toString();
	    }
	    if (col == 1) {
		return sme.getValue();
	    }
	    return null;
	}
	
	public boolean isCellEditable(int row, int col) {
	    if (col == 1) {
		return true;
	    }
	    return false;
	}

	public void refresh() {
	    search_metadata_entries = getEntries();
	    
	}
	
	public void setValueAt(Object new_value, int row, int col) {
	    SearchMetadataEntry sme = (SearchMetadataEntry) search_metadata_entries.get(row);
	    String old_value = sme.getValue();
	    if (!new_value.equals(old_value)) {
		sme.setValue((String)new_value);
		
	    }

	}

    }
    private static class SearchMetadataTableCellRenderer
	extends DefaultTableCellRenderer {

	public void setValue(Object value) {
	    
	    setText((String)value);
	}
	/** Returns the default table cell renderer.
	 * @param table The <strong>JTable</strong>.
	 * @param value The value to assign to the cell at [row, column] as an <strong>Object</strong>.
	 * @param isSelected <i>true</i> if cell is selected.
	 * @param hasFocus <i>true</i> if cell has focus.
	 * @param row The row of the cell to render as an <i>int</i>.
	 * @param column The column of the cell to render as an <i>int</i>.
	 * @return The default table cell renderer <strong>Component</strong>.
	 */
	public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
	    JComponent component = (JLabel) super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

	    // real_column is the column in the model, column is the column in the table - may be different if the user has moved the columns around
	    int real_column =  table.convertColumnIndexToModel(column);
	    if (real_column == 1 && isSelected) {
		table.editCellAt(row, column);
		if (table.isEditing()) {
		    table.getEditorComponent().requestFocus();
		}
	    }
	    
	    // so we can see the background colour
	    component.setOpaque(true);

	    // Background
	    if (isSelected) {
		component.setBackground(Configuration.getColor("coloring.workspace_heading_background", true));
	    }
	    else {
		if (real_column == 0) {
		    component.setBackground(Configuration.getColor("coloring.collection_heading_background", true));
		}
		else {
		    component.setBackground(Configuration.getColor("coloring.collection_tree_background", true));
		}
	    }
	    
	    // The value column of cells never paints focus
	    if (real_column == 1) {
		component.setBorder(BorderFactory.createEmptyBorder(0,2,0,0));
	    }
	 
	    return component;
	}

    }
    
}
