###########################################################################
#
# MetadataRead - like a Java interface that defines that a subclass is
# a Plugin that extracts Metadata
#
# A component of the Greenstone digital library software
# from the New Zealand Digital Library Project at the 
# University of Waikato, New Zealand.
#
# Copyright (C) 2008 New Zealand Digital Library Project
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
###########################################################################

package MetadataRead;

use PrintInfo;
use strict;

# MetadataRead is an abstract superclass that does not inherit from anything else.
# It exists solely to define the can_process_this_file_for_metadata() method in
# such a way that those MetadataPlugins that inherit from MetadataRead don't need 
# to define this method and will always process the files associated with them for 
# metadata and other plugins in the pipeline won't be passed these files anymore.

# MetadataRead defines method can_process_this_file_for_metadata() with identical
# signature to BaseImporter. (MetadataRead doesn't inherit from BaseImporter, so it's
# not 'overriding' it.) Subclasses of MetadataRead that want to use this method 
# definition can override their inherited BaseImporter version of the method by 
# listing MetadataRead as the *first* superclass they inherit from in the ISA list.
# This is the way Perl resolves conflicting method definitions.

my $arguments = [];

my $options = { 'name'     => "MetadataRead",
		'desc'     => "{MetadataRead.desc}",
		'abstract' => "yes",
		'inherits' => "yes",
		'args'     => $arguments };


sub new {
    my ($class) = shift (@_);
    my ($pluginlist,$inputargs,$hashArgOptLists,$auxiliary) = @_;
    push(@$pluginlist, $class);

    push(@{$hashArgOptLists->{"ArgList"}},@{$arguments});
    push(@{$hashArgOptLists->{"OptList"}},$options);

    # Like PrintInfo, MetadataRead has no superclass,
    # so $self is intialised to an empty array.
    my $self = {};
    return bless $self, $class;

}

# MetadataPlugins that inherit from MetadataRead will by default
# process all the metadata in files whose extensions match. 
# Override this method in a subclass to return undef if other 
# files should also be allowed to process the metadata therafter.
sub can_process_this_file_for_metadata {
    my $self = shift(@_);

#	print STDERR "********* MetadataRead::can_process_this_file_for_metadata() called.\n";
	
    return $self->can_process_this_file(@_);
}

# filename_for_metadata is the name of the file to attach metadata to. $new_metadata is a hash of all the metadata. file is the metadata file, filename_full_path is full path to metadata file
sub store_meta_in_extrametadata
{
     my $self = shift(@_);
     
     my ($filename_for_metadata, $new_metadata, $file, $filename_full_path,
	 $extrametakeys, $extrametadata, $extrametafile) = @_;

     # Extrametadata keys should be regular expressions
     # Indexing into the extrameta data structures requires the filename's style of slashes to be in URL format
     # Then need to convert the filename to a regex, no longer to protect windows directory chars \, but for
     # protecting special characters like brackets in the filepath such as "C:\Program Files (x86)\Greenstone".
     $filename_for_metadata = &util::filepath_to_url_format($filename_for_metadata);
    $filename_for_metadata = &util::filename_to_regex($filename_for_metadata);

    # Check that we haven't already got some metadata
    if (defined &extrametautil::getmetadata($extrametadata, $filename_for_metadata)) {
	print STDERR "\n****  MetadataRead: Need to merge new metadata with existing stored metadata: file = $filename_for_metadata\n" if $self->{'verbosity'} > 3;

	my $file_metadata_table = &extrametautil::getmetadata($extrametadata, $filename_for_metadata);

	foreach my $metaname (keys %{$new_metadata}) {
	    # will create new entry if one does not already exist
	    push(@{$file_metadata_table->{$metaname}}, @{$new_metadata->{$metaname}});	    
	}

    } else {
	&extrametautil::setmetadata($extrametadata, $filename_for_metadata, $new_metadata);
	&extrametautil::addmetakey($extrametakeys, $filename_for_metadata);
    }

     if (defined $file && defined $filename_for_metadata) {
	 #maps the file to full path
	&extrametautil::addmetafile($extrametafile, $filename_for_metadata, $file, $filename_full_path);
     }


#    }
}

1;
