/**********************************************************************
 *
 * nullproto.cpp -- 
 * Copyright (C) 1999  The New Zealand Digital Library Project
 *
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/

#include "nullproto.h"
#include <assert.h>

#include "filter.h"
#include "browsefilter.h"

#include "fileutil.h"
// for read_file

nullproto::nullproto() {
  cset = NULL;
}

nullproto::~nullproto() {
}

// add collection to collectset (this should not be called until after the
// collectset has been added to the protocol with set_collectset()
void nullproto::add_collection (const text_t &collection, void *recpt, 
				const text_t &gsdlhome, const text_t& collecthome,
				const text_t &dbhome) 
{      
  
  if (cset != NULL) {
    this->cset->add_collection (collection, gsdlhome, collecthome);
  }
}


// remove_collection deletes the collection server of collection.
// This only needs to be called if a collectionserver is to be
// removed while the library is running. The destructor function
// cleans up all collectservers when the program exits.
void nullproto::remove_collection (const text_t &collection, ostream &logout) {
  cset->remove_collection(collection, logout);
}

// this configure will configure each of the collection servers
void nullproto::configure (const text_t &key, const text_tarray &cfgline,
			   comerror_t &err) {
  // the naming of the collection should not be done here,
  // it should be done just after the collection server has been 
  // created

  // the protocol should not configure the collection set; it should be
  // done direct to the collection server set
  if (key == "gsdlhome" || key == "gdbmhome"
      || key == "collecthome" || key == "collectdir") {
    cset->configure(key, cfgline);
  }
}

// this init will configure and init each of the collection servers
bool nullproto::init (comerror_t &err, ostream &logout) {
  return cset->init(logout);
}

text_t nullproto::get_site_name (comerror_t &err) {
  return "localhost";
}

text_t nullproto::get_protocol_name (comerror_t &err) {
  return "nullproto";
}


void nullproto::get_collection_list (text_tarray &collist, comerror_t &err, 
				     ostream &/*logout*/) {
  cset->getCollectionList(collist);
  err = noError; // no error is guaranteed - collection server shouldn't
                 // about receptionist error states, logout is irrelevant
}

void nullproto::has_collection (const text_t &collection, bool &hascollection, 
				comerror_t &err, ostream &/*logout*/) {
  hascollection = (cset->getCollectServer(collection) != NULL);
  err = noError;
}

void nullproto::ping (const text_t &collection, bool &wassuccess, 
		      comerror_t &err, ostream &logout) {
  collectserver *cserver = cset->getCollectServer(collection);
  if (cserver != NULL) cserver->ping(wassuccess, err, logout);
  else err = protocolError;
}

void nullproto::get_collectinfo (const text_t &collection, 
				 ColInfoResponse_t &collectinfo,
				 comerror_t &err, ostream &logout) {
  collectserver *cserver = cset->getCollectServer (collection);
  if (cserver != NULL) cserver->get_collectinfo (collectinfo, err, logout);
  else err = protocolError;
}


void nullproto::get_filterinfo (const text_t &collection,
				InfoFiltersResponse_t &response,
				comerror_t &err, ostream &logout) {
  collectserver *cserver = cset->getCollectServer (collection);
  if (cserver != NULL) cserver->get_filterinfo (response, err, logout);
  else err = protocolError;
}

void nullproto::get_filteroptions (const text_t &collection,
				   const InfoFilterOptionsRequest_t &request,
				   InfoFilterOptionsResponse_t &response, 
				   comerror_t &err, ostream &logout) {
  collectserver *cserver = cset->getCollectServer (collection);
  if (cserver != NULL) cserver->get_filteroptions (request, response, err, logout);
  else err = protocolError;
}

// returns the contents of a collection's rss-items.rdf file (generated by BasePlugout)
void nullproto::get_rss_items (const text_t &collection, 
				const text_t &gsdlhome, 
				text_t &rss_items, 
				comerror_t &err, 
				ostream &logout) {

  text_t rss_filename = filename_cat(gsdlhome,"collect",collection,"index","rss-items.rdf");
  // read contents of file rss_filename into the string rss_items
  if (read_file(rss_filename,rss_items)) { 
    err = noError;
  } else {
    err = protocolError;
  }
}

void nullproto::filter (const text_t &collection,
			FilterRequest_t &request,
			FilterResponse_t &response,
			comerror_t &err, ostream &logout) {
  collectserver *cserver = cset->getCollectServer (collection);
  if (cserver != NULL) cserver->filter (request, response, err, logout);
  else err = protocolError;
}

void nullproto::get_document (const text_t &collection,
			      const DocumentRequest_t &request,
			      DocumentResponse_t &response,
			      comerror_t &err, ostream &logout) {
  collectserver *cserver = cset->getCollectServer (collection);
  if (cserver != NULL) cserver->get_document (request, response, err, logout);
  else err = protocolError;
}

void nullproto::is_searchable (const text_t &collection, bool &issearchable, 
			       comerror_t &err, ostream &logout) {
  issearchable = false;
  collectserver *cserver = cset->getCollectServer (collection);
  if (cserver != NULL) cserver->is_searchable (issearchable, err, logout);
  else err = protocolError;
}
