/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: John Thompson, Greenstone Digital Library, University of Waikato
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.cdm;

import org.greenstone.gatherer.Gatherer;
import org.greenstone.gatherer.Configuration;
import org.greenstone.gatherer.util.Codec;
import org.greenstone.gatherer.util.StaticStrings;
import org.greenstone.gatherer.util.XMLTools;
import org.w3c.dom.*;


/** This class encapsulates a single collection level metadata assignment, which constitutes a name, language and value.
 * @author John Thompson, Greenstone Digital Library, University of Waikato
 * @version 2.4
 */
public class CollectionMeta
    implements DOMProxyListEntry {

    static final public boolean TEXT = true;
    static final public boolean GREENSTONE = false;

    protected boolean dummy = false;
    protected Element element = null;
    protected String text = null;

    /** default no args constructor which gets called explicitly from subclass construcotrs */
    public CollectionMeta() {
    }
    /** Constructor.
     * @param element the Element from which we will determine metadata details
     */
    public CollectionMeta(Element element) {
	this.element = element;
    }

    /** Constructor to create a new piece of metadata given its name. */
    public CollectionMeta(String name) {
	element = CollectionConfiguration.createElement(StaticStrings.COLLECTIONMETADATA_ELEMENT);
	element.setAttribute(StaticStrings.NAME_ATTRIBUTE, name);
	element.setAttribute(StaticStrings.LANGUAGE_ATTRIBUTE, Configuration.getLanguage());
	element.setAttribute(StaticStrings.ASSIGNED_ATTRIBUTE, StaticStrings.FALSE_STR);
    }

    /** Constructor to create a new piece of metadata given its name. */
    public CollectionMeta(String name, String language) {
	element = CollectionConfiguration.createElement(StaticStrings.COLLECTIONMETADATA_ELEMENT);
	element.setAttribute(StaticStrings.NAME_ATTRIBUTE, name);
	element.setAttribute(StaticStrings.LANGUAGE_ATTRIBUTE, language);
	element.setAttribute(StaticStrings.ASSIGNED_ATTRIBUTE, StaticStrings.FALSE_STR);
    }

    /** Constructor to create a new piece of metadata given its name. */
    public CollectionMeta(String name, String language, boolean dummy) {
	this(name, language);
	this.dummy = dummy;
    }

    /** Method to compare two collection metadata objects to calculate their respective ordering.
     * @param object the other metadata to compare to, as an Object
     * @return an int which is less than 0 if this object proceeds the given object, 0 if they are equal and greater than 0 otherwise.
     * @see org.greenstone.gatherer.cdm.Language
     */
    public int compareTo(Object object) {
	return toString().compareTo(object.toString());
    }

    /** Factory constructor. */
    public DOMProxyListEntry create(Element element) {
	return new CollectionMeta(element);
    }

    /** Method to compare two collection metadata objects for equality.
     * @param object The other metadata to compare to, as an <strong>Object</strong>.
     * @return A <i>boolean</i> value of <i>true</i> if the object are equal, <i>false</i> otherwise.
     */
    public boolean equals(Object object) {
	return (compareTo(object) == 0);
    }

    public Element getElement() {
	return element;
    }

    /** Method to retrieve the value of language.
     * @return the value of language as a <strong>String</strong>.
     */
    public String getLanguage() {
	// Retrieve the language string
	return element.getAttribute(StaticStrings.LANGUAGE_ATTRIBUTE);
    }

    /** Method to retrieve the value of name.
     * @return the name attribute of the collection meta as a String 
     */
    public String getName() {
	return element.getAttribute(StaticStrings.NAME_ATTRIBUTE);
    }
    /** Method to retrieve the value of value (well great choice of name there).
     * @return The value of value as a <strong>String</strong>.
     */
    public String getValue(boolean text_value) {
	String raw_value = XMLTools.getValue(element);
	// Decode the raw value depending on whether the user asked for the TEXT or GREENSTONE version
	if(text_value == TEXT) {
	    return Codec.transform(raw_value, Codec.DOM_TO_TEXT);
	}
	else {
	    return Codec.transform(raw_value, Codec.DOM_TO_GREENSTONE);
	}
    }

    public boolean isAssigned() {
	return (element != null && !element.getAttribute(StaticStrings.ASSIGNED_ATTRIBUTE).equals(StaticStrings.FALSE_STR));
    }

    public boolean isDummy() {
	return dummy;
    }

    /** Determine if this metadata is one of the four special pieces of metadata.
     * @return true if this metadata is special, false otherwise.
     */
    public boolean isSpecial() {
	return (element != null && element.getAttribute(StaticStrings.SPECIAL_ATTRIBUTE).equals(StaticStrings.TRUE_STR));
    }

    public void setAssigned(boolean assigned) {
	if(element != null) {
	    element.setAttribute(StaticStrings.ASSIGNED_ATTRIBUTE, (assigned ? StaticStrings.TRUE_STR : StaticStrings.FALSE_STR));
	}
    }

    public void setElement(Element element) {
	this.element = element;
	text = null;
    }

    /** Change the value of value. 
     * @param raw_value the new value as a String.
     */
    public void setValue(String raw_value) {
	setValue(raw_value, false);
    }

    public void setValue(String raw_value, boolean preserveTags) {
	// we need to check if the value has changed
	String current_value = XMLTools.getValue(element);
	
	String new_value = preserveTags ?
	    Codec.transform(raw_value, Codec.TEXT_TO_DOM_PRESERVE_TAGS)
	    : Codec.transform(raw_value, Codec.TEXT_TO_DOM);
	if (!current_value.equals(new_value)) {
	    // Only raw html text can be given to setValue so we need to encode it
	    XMLTools.setValue(element, new_value);
	    text = null; // Reset text
	    // And determine if this makes the metadata assigned
	    setAssigned(raw_value != null && raw_value.length() > 0);
	}
    }


    /** Method to print out this class as it would appear within the collection configuration file.
     * @return A <strong>String</strong> containing the text value of this class.
     */
    public String toString() {
	if(text == null) {
	  text = CollectCfgReadWrite.toString(element);
	    if (text.equals("")){
		text = getName();
	    }
	}
	return text;
    }
}
