/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: Greenstone Digital Library, University of Waikato
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.download;

import java.net.InetSocketAddress;
import java.net.Proxy;
import java.net.URL;
import java.net.URLConnection;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import java.security.cert.X509Certificate;


/** Use this class to get a URL connection with or without proxy settings, for http or https URLS,
 * and with no-check-certificate on or off for https URLs.
 * This class has the ability to turn off checking security certificates for https URLs to allow us to do 
 * the Java equivalent to running wget with --no-check-certificate. That part of the code is from
 * https://stackoverflow.com/questions/6659360/how-to-solve-javax-net-ssl-sslhandshakeexception-error
 * http://www.nakov.com/blog/2009/07/16/disable-certificate-validation-in-java-ssl-connections/
*/ 
public class URLConnectionManager implements HostnameVerifier, X509TrustManager 
{
	// Create a reusable trust manager that does not validate certificate chains and considers all hosts valid
	private static final URLConnectionManager allTrustingSSLTrustManager = new URLConnectionManager();
    private static final TrustManager[] trustAllCerts = new TrustManager[] {allTrustingSSLTrustManager};

	// save original defaults, in case we want to start restoring these in future
	private static final HostnameVerifier restoreDefaultHostnameVerifier = HttpsURLConnection.getDefaultHostnameVerifier(); 
	private static final SSLSocketFactory restoreDefaultSSLSocketFactory = HttpsURLConnection.getDefaultSSLSocketFactory();
	
	//********** THE METHODS TO USE TO GET A URL CONNECTION *************//
	//********** BASED ON PROXY, URL PROTOCOL (HTTPS) AND IF NO_CHECK_CERTIFICATES IS TO BE ON OR OFF FOR HTTPS *************//
	public static void setNoCheckCertificates(boolean noCheckCertificates) throws Exception {
		if(noCheckCertificates) {
			// Install the all-trusting trust manager
			SSLContext sc = SSLContext.getInstance("SSL");
			sc.init(null, trustAllCerts, new java.security.SecureRandom());
			HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());
			HttpsURLConnection.setDefaultHostnameVerifier(allTrustingSSLTrustManager);
		} else {
			HttpsURLConnection.setDefaultSSLSocketFactory(restoreDefaultSSLSocketFactory);
			HttpsURLConnection.setDefaultHostnameVerifier(restoreDefaultHostnameVerifier);
		}
	}
	
	public static URLConnection getConnection(String url_str, Proxy proxy) throws Exception {
		// use existing settings for noCheckCertificates -- caller would have called setNoCheckCertificates to set this up at some time in the past
		
		URL url = new URL(url_str);
		// if we're given a proxy to access the URL with, use it
		if(proxy == null) {
			return url.openConnection(); 
		} else {
			return url.openConnection(proxy);
		}
	}
	
	public static URLConnection getConnection(String url_str, Proxy proxy, boolean noCheckCertificates) throws Exception {
		URL url = new URL(url_str);
		if(url_str.startsWith("https:") && noCheckCertificates) { // requested to turn off certificate validation for HTTPs URLS
			setNoCheckCertificates(noCheckCertificates);
					
		} else { // no certificate checking needed if noCheckCertificates is false or if the url is anything other than HTTPS
			setNoCheckCertificates(false);
		}
		
		// if we're given a proxy to access the URL with, use it
		if(proxy == null) {
			return url.openConnection(); 
		} else {
			return url.openConnection(proxy);
		}
	}
	
	public static URLConnection getConnection(String url_str, String proxy_host, String proxy_port, boolean noCheckCertificates) throws Exception {
		Proxy proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxy_host, Integer.parseInt(proxy_port)));
		return getConnection(url_str, proxy, noCheckCertificates);
	}
	
	public static URLConnection getConnection(String url_str, String proxy_host, int proxy_port, boolean noCheckCertificates) throws Exception {
		Proxy proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxy_host, proxy_port));
		return getConnection(url_str, proxy, noCheckCertificates);
	}
	
	//********************** IMPLEMENTED METHODS **********************//	

	//*********** interface X509TrustManager *************//
	public java.security.cert.X509Certificate[] getAcceptedIssuers() {
		return null;
	}
	public void checkClientTrusted(X509Certificate[] certs, String authType) {}
	public void checkServerTrusted(X509Certificate[] certs, String authType) {}
	
	//*********** interface HostnameVerifier *************//
	// all hosts are considered valid
	public boolean verify(String hostname, SSLSession session) {
		return true;
	}
	
}
