/**
 *############################################################################
 * A component of the Greenstone Librarian Interface, part of the Greenstone
 * digital library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: Michael Dewsnip, NZDL Project, University of Waikato, NZ
 *
 * Copyright (C) 2004 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *############################################################################
 */

package org.greenstone.gatherer.metadata;


import java.io.*;
import java.util.*;
import org.greenstone.gatherer.Gatherer;
import org.greenstone.gatherer.collection.CollectionManager;
import org.greenstone.gatherer.remote.RemoteGreenstoneServer;
import org.greenstone.gatherer.util.XMLTools;
import org.w3c.dom.*;


/** This class represents one profile.xml file */
public class ProfileXMLFile
    extends File
{
    private HashMap metadata_mapping = null;


    public ProfileXMLFile(String profile_xml_file_path)
    {
	super(profile_xml_file_path);
    }


    public String getMetadataElementFor(String metadata_element_name_full)
    {
	if (getMetadataMapping() == null) {
	    return null;
	}

	return (String) metadata_mapping.get(metadata_element_name_full);
    }


    public HashMap getMetadataMapping()
    {
	// If we have already loaded the metadata mapping, use it
	if (metadata_mapping != null) {
	    return metadata_mapping;
	}

	// Build the metadata mapping from the profile.xml file
	metadata_mapping = new HashMap();

	// Parse the profile.xml file
	Document document = XMLTools.parseXMLFile(this);
	if (document == null) {
	    System.err.println("Error: Could not parse profile.xml file " + getAbsolutePath());
	    return null;
	}

	// Read all the Action elements in the file
	NodeList action_elements_nodelist = document.getElementsByTagName("Action");
	for (int i = 0; i < action_elements_nodelist.getLength(); i++) {
	    Element current_action_element = (Element) action_elements_nodelist.item(i);
	    String source_metadata_element_name_full = current_action_element.getAttribute("source");
	    String target_metadata_element_name_full = current_action_element.getAttribute("target");
	    metadata_mapping.put(source_metadata_element_name_full, target_metadata_element_name_full);
	}

	return metadata_mapping;
    }


    public void mapElement(String source_metadata_element_name_full, String target_metadata_element_name_full)
    {
	// Parse the profile.xml file
	Document document = XMLTools.parseXMLFile(this);
	if (document == null) {
	    System.err.println("Error: Could not parse profile.xml file " + getAbsolutePath());
	    return;
	}

	// Create a new Action element to record this mapping
	Element new_action_element = document.createElement("Action");
	new_action_element.setAttribute("source", source_metadata_element_name_full);
	new_action_element.setAttribute("target", target_metadata_element_name_full);
	document.getDocumentElement().appendChild(new_action_element);

	// Rewrite the profile.xml file
	XMLTools.writeXMLFile(this, document);

	// This is inefficient but for simplicity we'll just upload the file every time it is changed
	if (Gatherer.isGsdlRemote) {
	    Gatherer.remoteGreenstoneServer.uploadCollectionFile(CollectionManager.getLoadedCollectionName(), this);
	}

	// Invalidate the metadata mapping
	metadata_mapping = null;
    }
}
