/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: John Thompson, Greenstone Digital Library, University of Waikato
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.cdm;

import org.greenstone.gatherer.util.StaticStrings;
import org.greenstone.gatherer.util.XMLTools;
import org.w3c.dom.*;


/** This class encapsulates one subcollection entry in the collection configuration file.
 * @author John Thompson, Greenstone Digital Library, University of Waikato
 * @version 2.4
 */
public class Subcollection
    implements Comparable, DOMProxyListEntry {
    /** A <i>boolean</i> which is <i>true</i> if the condition is an include one, <i>false</i> otherwise. */
    private boolean include = true;
    /** The DOM Element this Subcollection is based upon. */
    private Element element = null;
    /** A String containing a Perl expression which is used as the pattern by which to filter this subcollection. */
    private String pattern = null;
    /** A series of flags to be used when matching the expression. */
    private String flags = null;
    /** A String which is a unique identifier of a subcollection. */
    private String name = null;
    /** Either the fully qualified name of the metadata whose value should be matched against the given expression, or <i>null</i> if you wish to match against the file name. */
    private String source = null;
    private String text = null;

    /** Default constructor which should only be used during DOMProxyListModel initialization. */
    public Subcollection() {
    }

    /** Constructor for representing an existing assigned Subcollection.
     * @param element the Element this subcollection is based upon
     */
    public Subcollection(Element element) {
	this.element = element;
    }

    /** Constructor for assigning a brand new Subcollection.
     * @param name a unique identifier for this collection as a String
     * @param include true if this the pattern should be an inclusion filter, false for exclusion
     * @param source either the fully qualified name of an element as a String, or null to filter filenames
     * @param pattern the matching pattern as a String
     * @param flags any flags to use while matching, as a String
     */
    public Subcollection(String name, boolean include, String source, String pattern, String flags) {
	// Cache the details
	this.flags = flags;
	this.include = include;
	this.name = name;
	this.pattern = pattern;
	if(source != null) {
	    this.source = source;
	}
	else {
	    this.source = StaticStrings.FILENAME_STR;
	}
	// Create a new DOM Element with the appropriate attributes and text value
	element = CollectionConfiguration.createElement(StaticStrings.SUBCOLLECTION_ELEMENT);
	element.setAttribute(StaticStrings.CONTENT_ATTRIBUTE, source);
	element.setAttribute(StaticStrings.NAME_ATTRIBUTE, name);
	element.setAttribute(StaticStrings.OPTIONS_ATTRIBUTE, flags);
	element.setAttribute(StaticStrings.TYPE_ATTRIBUTE, (include ? StaticStrings.INCLUDE_STR : StaticStrings.EXCLUDE_STR));
	XMLTools.setValue(element, pattern);
    }

    /** Method to compare two subcollections.
     * @param object he other subcollection to compare to, as an Object
     * @return an int which is &gt;0, 0, or &lt;0 if this subcollection is before, equal to, or after the target object respectively.
     */
    public int compareTo(Object object)
    {
	if (object == null) {
	    return -1;
	}

	// Object may be a String
	if (object instanceof String) {
	    return getName().compareTo((String) object);
	}

	// Otherwise assume it is a Subcollection object
	return getName().compareTo(((Subcollection) object).getName());
    }


    public DOMProxyListEntry create(Element element) {
	return new Subcollection(element);
    }

    /** Method to check two subcollections for equality.
     * @param object the other subcollection to compare to, as an Object
     * @return true if the subcollections are equal, false otherwise.
     */
    public boolean equals(Object object) {
	return (compareTo(object) == 0);
    }

    /** Retrieve the DOM Element this Subcollection is based upon.
     * @return an Element
     */
    public Element getElement() {
	return element;
    }

    /** Method to get the value of flags.
     * @return the value of flags as a String
     */
    public String getFlags() {
	if(flags == null && element != null) {
	    flags = element.getAttribute(StaticStrings.OPTIONS_ATTRIBUTE);
	}
	return flags;
    }

    /** Method to get the value of name.
     * @return String The value of name as a <strong>String</string>.
     */
    public String getName() {
	if(name == null && element != null) {
	    name = element.getAttribute(StaticStrings.NAME_ATTRIBUTE);
	}
	return name;
    }

    public String getPattern() {
	if(pattern == null && element != null) {
	    pattern = XMLTools.getValue(element);
	}
	return pattern;
    }

    /** Method to get the name of the source of the strings used in pattern matching.
     * @return a String which is either the fully qualified name of a metadata element, or filename
     */
    public String getSource()
    {
	if (source == null && element != null) {
	    source = element.getAttribute(StaticStrings.CONTENT_ATTRIBUTE);
	}

	return source;
    }

    public boolean isAssigned() {
	return (element != null && !element.getAttribute(StaticStrings.ASSIGNED_ATTRIBUTE).equals(StaticStrings.FALSE_STR));
    }

    /** Method to get the value of include.
     * @return boolean true if this is an inclusion filter, false otherwise
     */
    public boolean isInclusive() {
	if (element != null) {
	    include = element.getAttribute(StaticStrings.TYPE_ATTRIBUTE).equals(StaticStrings.INCLUDE_STR);
	}

	return include;
    }

    public void setAssigned(boolean assigned) {
	if(element != null) {
	    element.setAttribute(StaticStrings.ASSIGNED_ATTRIBUTE, (assigned ? StaticStrings.TRUE_STR : StaticStrings.FALSE_STR));
	}
    }

    public void setElement(Element element) {
	this.element = element;
	include = true;
	flags = null;
	name = null;
	pattern = null;
	source = null;
	text = null;
    }

    public void setFlags(String new_flags) {
	if(element != null) {
	    // Change element
	    element.setAttribute(StaticStrings.OPTIONS_ATTRIBUTE, new_flags);
	    flags = new_flags;
	    text = null;
	}
    }

    public void setInclusive(boolean new_include) {
	if(element != null) {
	    // Change element
	    include = new_include;
	    element.setAttribute(StaticStrings.TYPE_ATTRIBUTE, (include ? StaticStrings.INCLUDE_STR : StaticStrings.EXCLUDE_STR));
	    text = null;
	}
    }

    public void setName(String new_name) {
	if(element != null) {
	    // Change element
	    element.setAttribute(StaticStrings.NAME_ATTRIBUTE, new_name);
	    name = new_name;
	    text = null;
	}
    }

    public void setPattern(String new_pattern) {
	if(element != null) {
	    // Change element
	    XMLTools.setValue(element, new_pattern);
	    pattern = new_pattern;
	    text = null;
	}
    }

    public void setSource(String new_source) {
	if(element != null) {
	    // Change element
	    element.setAttribute(StaticStrings.CONTENT_ATTRIBUTE, new_source);
	    source = new_source;
	    text = null;
	}
    }

    /** Method to display the contents of this class as it would appear in the collection configuration file.
     * @return a String representing this subcollection
     */
    public String toString() {
	if(text == null) {// && element != null) {

	    StringBuffer new_text = new StringBuffer(getName());
	    new_text.append(StaticStrings.SPACE_CHARACTER);
	    new_text.append(StaticStrings.SPEECH_CHARACTER);
	    if (!isInclusive()) {
		new_text.append(StaticStrings.EXCLAMATION_CHARACTER);
	    }
	    new_text.append(getSource());
	    new_text.append(StaticStrings.SEPARATOR_CHARACTER);
	    new_text.append(getPattern());
	    new_text.append(StaticStrings.SEPARATOR_CHARACTER);
	    new_text.append(getFlags());
	    new_text.append(StaticStrings.SPEECH_CHARACTER);
	    text =  new_text.toString();
	}
	
	return text;
    }
}
