/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: John Thompson, Greenstone Digital Library, University of Waikato
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.file;

import javax.swing.tree.*;
import org.greenstone.gatherer.util.DragComponent;

/** This object encapsulates all the information about a certain file movement job. This job be to create a copy of a file from one place to another, or to simply delete a file. 
 * @author John Thompson
 * @version 2.3c
 */
public class FileJob {
    /** true to mark that this file has already been copied, false otherwise. */
    public boolean done = false;

    /** The type of this movement as an byte. */
    public byte type = 0;
    /** The DragComponent source of this file, most likely a GTree. */
    public DragComponent source       = null;
    /** The DragComponent to move the file to, again most likely a GTree. */
    public DragComponent target       = null;
    /** The unique identifier shared by all jobs created by the same action. */
    private long id = 0;
    /** The path to the destination node as a string. Used because any reference based path or variables quickly becomes obsolete. */
    private TreePath destination_path = null;
    /** The path to the origin node as a string. Used because any reference based path or variables quickly becomes obsolete. */
    private TreePath origin_path = null;
    /** An element of the job type enumeration indicating a copy action. */
    static final public byte COPY   = 1;
    /** An element of the job type enumeration indicating a delete action. */
    static final public byte DELETE = 2;
    /** An element of the job type enumeration indicating a move action. */
    static final public byte MOVE   = 3;
    /** An element of the job type enumeration indicating a new folder action. */
    static final public byte NEW_FOLDER = 4;
    /** An element of the job type enumeration indicating an empty folder delete action. */
    static final public byte DELETE_EMPTY_DIRECTORY = 5;
    /** An element of the job type enumeration indicating a rename action. */
    static final public byte RENAME = 6;
    /** An element of the job type enumeration indicating a replace action. */
    static final public byte REPLACE = 7;
    /** An element of the job type enumeration indicating a copy action (but don't look for metadata. */
    static final public byte COPY_FILE_ONLY   = 8;

    /** Constructor.
     * @param id A unique identifier for this job (and others created with a single gesture) as a long.
     * @param source The DragComponent source of this file, most likely a GTree.
     * @param orig The FileNode you wish to mode.
     * @param target The DragComponent to move the file to, again most likely a GTree.
     * @param dest The files new FileNode parent within the target.
     * @param type The type of this movement as an int, either COPY or DELETE.
     */
    public FileJob(long id, DragComponent source, FileNode orig, DragComponent target, FileNode dest, byte type) {
	this.id = id;
	this.source = source;
	this.target = target;
	this.type = type;
	///ystem.err.println("New Job: " + type + ", " + source + ", " + target);
	// Dont store FileNodes which can go stale. Store paths instead, which are used to locate current 'fresh' versions of nodes.
	if(dest != null) {
	    this.destination_path = new TreePath(dest.getPath());
	    ///ystem.err.println("Destination Path: " + destination_path);
	}
	if(orig != null) {
	    this.origin_path = new TreePath(orig.getPath());
	    ///atherer.println("Origin Path: " + origin_path);
	}
    }

    /** Retrieve the destination node. Watch out for stale versions by always attempting to load the node at destination_path first. */
    public FileNode getDestination() {
	FileNode destination = null;
	if(destination_path != null) {
	    if(target != null) {
		FileSystemModel model = (FileSystemModel)target.getTreeModel();
		destination = model.getNode(destination_path);
	    }
	    // If the above fails, a stale copy may be better than nothing.
	    else {
		destination = (FileNode) destination_path.getLastPathComponent();
	    }
	}
	return destination;
    }

    /** Retrieve the origin node. Watch out for stale versions by always attempting to load the node at origin_path first. */
    public FileNode getOrigin() {
	FileNode origin = null;
	if(origin_path != null) {
	    if(source != null) {
		FileSystemModel model = (FileSystemModel)source.getTreeModel();
		origin = model.getNode(origin_path);
	    }
	    // If the above fails, a stale copy may be better than nothing.
	    else {
		origin = (FileNode) origin_path.getLastPathComponent();
	    }
	}
	return origin;
    }

    /** Retrieve the id for this job. */
    public long ID() {
	return id;
    }

    public String toString() {
	StringBuffer text = new StringBuffer("");
	switch(type) {
	case COPY:
	    text.append("copy ");
	    break;
	case DELETE:
	    text.append("delete ");
	    break;
	case MOVE:
	    text.append("move ");
	    break;
	case DELETE_EMPTY_DIRECTORY:
	    text.append("empty directory delete ");
	    break;
	default:
	    text.append("unknown ");
	}
	FileNode origin = getOrigin();
	if(origin != null) {
	    text.append(origin.getFile().getAbsolutePath());
	}
	else {
	    text.append("ERROR!");
	}
	if (type != DELETE && type != DELETE_EMPTY_DIRECTORY) {
	    text.append(" -> ");
	    FileNode destination = getDestination();
	    if(destination != null) {
		text.append(destination.getFile().getAbsolutePath());
	    }  
	    else {
		text.append("Recycle Bin");
	    }
	}
	return text.toString();
    }
}
