/**
 *############################################################################
 * A component of the Greenstone Librarian Interface, part of the Greenstone
 * digital library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: Michael Dewsnip, NZDL Project, University of Waikato, NZ
 *
 * Copyright (C) 2004 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *############################################################################
 */

package org.greenstone.gatherer.metadata;


import java.io.*;
import java.util.*;
import org.greenstone.gatherer.DebugStream;
import org.greenstone.gatherer.util.Utility;

/** This class is a static class that manages the doc.xml files */
public class DocXMLFileManager
{
    static private ArrayList doc_xml_files = new ArrayList();

    static public void clearDocXMLFiles()
    {
	doc_xml_files.clear();
    }


    static public ArrayList getMetadataExtractedFromFile(File file)
    {
	// Work out relative file path and its hex encoded value here,
	// avoids making DocXMLFile.java recalculate these each time
	String file_relative_path = file.getAbsolutePath();
	int import_index = file_relative_path.indexOf("import");
	if (import_index != -1) {
	    file_relative_path = file_relative_path.substring(import_index + "import".length() + 1);
	}
	
	// Build up a list of metadata values extracted from this file
	ArrayList metadata_values = new ArrayList();

	// Look at each loaded doc.xml file to see if any have extracted metadata for this file
	for (int i = 0; i < doc_xml_files.size(); i++) {
	    DocXMLFile doc_xml_file = (DocXMLFile) doc_xml_files.get(i);
	    ///System.err.println("@@@@ Looking at doc.xml file: " + doc_xml_files.get(i));
	    metadata_values.addAll(doc_xml_file.getMetadataExtractedFromFile(file, file_relative_path));
	}

	return metadata_values;
    }


    static public void loadDocXMLFiles(File directory, String filename_match)
    {
	// Make sure the directory (archives) exists
	if (directory.exists() == false) {
	    return;
	}

	// Look recursively at each subfile of the directory for doc.xml files
	File[] directory_files = directory.listFiles();
	for (int i = 0; i < directory_files.length; i++) {
	    File child_file = directory_files[i];
	    if (child_file.isDirectory()) {
		loadDocXMLFiles(child_file,filename_match); 
	    }
	    else if (child_file.getName().equals(filename_match)) {
		// e.g. doc.xml (for regular Greenstone, docmets.xml for Fedora)

		loadDocXMLFile(child_file,filename_match);
	    }
	}
    }


    static private void loadDocXMLFile(File doc_xml_file_file,String filename_match)
    {
	String file = doc_xml_file_file.getAbsolutePath();

	// Need to do typecasts in the following to keep Java 1.4 happy
	DocXMLFile doc_xml_file 
	    = (filename_match.equals("docmets.xml")) 
	    ? (DocXMLFile) new DocMetsXMLFile(file) 
	    : (DocXMLFile) new DocGAFile(file);

	try {	    
	    doc_xml_file.skimFile();
	    doc_xml_files.add(doc_xml_file);
	}
	catch (Exception exception) {
	    // We catch any exceptions here so errors in doc.xml files don't stop the collection from loading
	    System.err.println("Error: Could not skim doc.xml file " + doc_xml_file.getAbsolutePath());
	    DebugStream.printStackTrace(exception);
	}
    }
}
