/**
 *#########################################################################
 * DigitalLibraryServicesAPIA.java - part of the demo-client for 
 * Greenstone 3, of the Greenstone digital library suite from the New 
 * Zealand Digital Library Project at the  * University of Waikato, 
 * New Zealand.
 * <BR><BR>
 * Copyright (C) 2008 New Zealand Digital Library Project
 * <BR><BR>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * <BR><BR>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *########################################################################
 */

package org.greenstone.gs3client.dlservices;

import java.util.Map;
import org.greenstone.gs3client.data.DocumentNodeData;

/** Interface that defines the digital library methods that need to be
 * provided by a digital library that's going to be incorporated into the 
 * Greenstone Java-Client, so that the client may be able to use/access 
 * its functionality. 
 * 
 * Note that the Strings returned by all defined methods -- with the exception  
 * of getLanguage() -- represent Greenstone3 XML response messages, as defined 
 * in the Greenstone3 manual. 
 * These methods do not throw exceptions, but rather embed any error information
 * in the XML Strings returned, as is done in Greenstone3 XML response 
 * messages. 
 * 
 * The final 'A' of DigitalLibraryServicesAPIA stands for Access, as the  
 * methods defined here are to give access to a digital library's repository. 
 * @author ak19
*/
public interface DigitalLibraryServicesAPIA {
	/** The properties file containing the initial digital library connection
	 * settings which get displayed in the connection dialog fields */
	public static final java.io.File propertiesFile 
		= new java.io.File("gs3democlient.properties");
	
	/** If there's a dialog, then the CancelException can be used to indicate
	 * that the user cancelled out of it rather than having seen it through.
	*/
	public class CancelException extends Exception {
		public CancelException() { super(); }
		public CancelException(String message) { super(message); }
		public CancelException(Throwable cause) { super(cause); }
		public CancelException(String message, Throwable cause) { 
			super(message, cause); 
		}
	}
	
	//public static String displayName = "<insert DL name>";
	
	/** @return the name of this digital library for displaying in the client */
	public String getDisplayName();
	
	/*@return the URL of the server - this would generally form the basis of the 
	 * overall url location of where associated files are stored. */
	//public String getBaseURL();
	
	/** @return the overall directory path for associated files (not including
	 * the document's nodeID/pid). This can be used to formulate the base url of 
	 * JEditorPane's HTML documents. If the documents from any digital library 
	 * do not contain relative paths, or otherwise deal with the resolution of
	 * relative urls themselves, then return "" from here. */
	public String getAssocFileBaseURL();
	
	/* @return any string prefixed to associated files like images. This will
	 * generally be "", but in Fedora's case, associated images of the 
	 * Greenstone documents stored in there start with "FG". */
	//public String getAssocFilePrefix();
	
	/** @return the directory path to the associated files of the given document 
	 * node. For instance, the base url of a JEditorPane's HTML documents can be 
	 * set to this. */
	public String getAssocFileBaseURL(DocumentNodeData docNode);
	
	/** Sets the preferred language (code) for metadata and other content to be
	 * returned on request messages. If the preferred language is not available,
	 * the implementation may choose to send English back instead, or failing
	 * that, send the data in any other available language. 
	 * @param language has to be a language code as recognised by Greenstone3
	 * (the language codes used by W3C probably). E.g. "en" for English. */
	public void setLanguage(String language);
	
	/** Gets the language that's set as the preferred language. 
	 * @return the language code as recognised by Greenstone3 (which are the 
	 * language codes used by W3C probably). E.g. "en" for English. */
	public String getLanguage();

	/** @return Greenstone3 XML describe response message with information
	 * about collections contained and services (and any serviceRacks) supported
	 * by the Digital library. */
	public String describe();
	
	/** @return Greenstone3 XML describe response message originating from a
	 * collection, describing the collection (display items) as well as
	 * listing any services supported specifically by the collection. 
	 * @param collection is the name of the collection to be described. */
	public String describeCollection(String collection);
	
	/** @return Greenstone3 XML describe response message originating from a
	 * collection's service, describing the service (display items). 
	 * @param collection is the name of the collection to be described
	 * @param service is the name of the collection's service to be described. */
	public String describeCollectionService(String collection, String service);
	
	/** @return Greenstone3 XML describe response message originating from a
	 * general (non-collection specific) service, describing the requested 
	 * service (for example with GS3 display items). 
	 * @param service is the name of the collection's service to be described */
	public String describeService(String service);

	/**	@return a String representing Greenstone3 DocumentMetadataRetrieve XML 
	 * containing all the metadata of all the documents indicated by docIDs 
	 * @param collection is the name of the collection 
	 * @param docIDs is an array of document identifiers of documents whose  
	 * metadata is requested 
	 * @param metadata is the list of metadata elements to be retrieved for each doc */
	public String retrieveDocumentMetadata(String collection, String[] docIDs, String[] metadata);
	
	/**	@return a String representing Greenstone3 DocumentMetadataRetrieve XML 
	 * containing all the metadata of the document denoted by docID 
	 * @param collection is the name of the collection 
	 * @param docID is the document identifier of the document whose metadata is 
	 * requested 
	 * @param metadata is the list of metadata elements to be retrieved for each doc */
    public String retrieveDocumentMetadata(String collection, String docID, String[] metadata);

	/**	@return a String representing Greenstone3 DocumentMetadataRetrieve XML 
	 * containing only the title metadata of the documents denoted by docIDs 
	 * @param collection is the name of the collection 
	 * @param docIDs is an array of document identifiers of documents whose titles 
	 * are requested */
	
	public String retrieveTitleMetadata(String collection, String[] docIDs);
	/**	@return a String representing Greenstone3 DocumentMetadataRetrieve XML 
	 * containing only the title metadata of the document denoted by docID 
	 * @param collection is the name of the collection 
	 * @param docID is the document identifier of the document whose titles is 
	 * requested */
	public String retrieveTitleMetadata(String collection, String docID);
	
	/**	@return a String representing Greenstone3 DocumentContentRetrieve XML 
	 * containing the document contents of the documents indicated by docIDs 
	 * @param collection is the name of the collection 
	 * @param docIDs is an array of document identifiers of documents whose (text)
	 * contents are requested */
	public String retrieveDocumentContent(String collection, String[] docIDs);
	
	/**	@return a String representing Greenstone3 DocumentContentRetrieve XML 
	 * containing the document contents of the document indicated by docID 
	 * @param collection is the name of the collection 
	 * @param docID is the document identifier of the document whose (text)
	 * content is requested */
	public String retrieveDocumentContent(String collection, String docID);

	/**	@return a String representing Greenstone3 DocumentMetadataRetrieve XML
	 * containing the document structure of the documents indicated by docIDs: 
	 * this means all their descendents 
	 * @param collection is the name of the collection 
	 * @param docIDs is an array of document identifiers of documents whose
	 * hierarchical structures are requested */
	public String retrieveDocumentStructure(String collection, String[] docIDs);
	
	/**	@return a String representing Greenstone3 DocumentMetadataRetrieve XML 
	 * containing the document structure of the document indicated by docID: 
	 * this means all its descendents 
	 * @param collection is the name of the collection 
	 * @param docID is the document identifier of the document whose hierarchical
	 * structure is requested */
	public String retrieveDocumentStructure(String collection, String docID); 


    	/** @return a String representing Greenstone3 DocumentMetadataRetrieve XML
	 * containing the requested portion of the document structure of the documents 
	 * indicated by docIDs: 
	 * @param collection is the name of the collection 
	 * @param docIDs is an array of document identifiers of documents whose
	 * hierarchical structures are requested 
	 * @param structure - strings specifying the required structure of each document. 
	 * It can be a combination of: ancestors, parent, siblings, children, descendants, entire.
	 * @param info - strings specifying the required structural info of each document. 
	 * It can be any combination of: siblingPosition, numSiblings, numChildren.
	*/
    public String retrieveDocumentStructure(String collection, String[] docIDs, String[] structure, String[] info);


    	/** @return a String representing Greenstone3 DocumentMetadataRetrieve XML
	 * containing the requested portion of the document structure of the documents 
	 * indicated by docIDs: 
	 * @param collection is the name of the collection 
	 * @param docID is the document identifier of the document whose hierarchical
	 * structure is requested
	 * @param structure - strings specifying the required structure of the document. 
	 * It can be a combination of: ancestors, parent, siblings, children, descendants, entire.
	 * @param info - strings specifying the required structural info of the document. 
	 * It can be any combination of: siblingPosition, numSiblings, numChildren.
	*/
    public String retrieveDocumentStructure(String collection, String docID, String[] structure, String[] info);


	// UNUSED by the client, but still a very useful method:
	/**	@return a String representing Greenstone3 DocumentMetadataRetrieve XML 
	 * containing a view of the document structure of the documents denoted by 
	 * docs where only the requested documents and their direct children are  
	 * returned. 
	 * @param collection is the name of the collection 
	 * @param docIDs is an array of document identifiers of documents whose
	 * hierarchical structures are requested */
	public String retrieveDocumentChildren(String collection, String[] docIDs);
	
	/**	@return a String representing Greenstone3 DocumentMetadataRetrieve XML 
	 * containing a view of the document structure of the document denoted by 
	 * docID where only the document and its direct children are returned. 
	 * @param collection is the name of the collection 
	 * @param docID is an document identifier of the document whose hierarchical
	 * structure is requested */
	public String retrieveDocumentChildren(String collection, String docID);
	
	/**	@return a String representing Greenstone3 ClassifierBrowse XML 
	 * giving the entire *structure* of the classification denoted by 
	 * classifierID (including the structures of document descendents of 
	 * the classifier). 
	 * @param classifierIDs - each ID is of the form CL# where the number (#)
	 * marks out structured sections like CL1.1.3 or CL2 
	 * @param collection is the name of the collection 
	 * @param service is the name of the browse service (=ClassifierBrowse usually) 
	 * @param structure - the requested browse substructure. Can be any combination
	 * of ancestors, parent, siblings, children, descendants.
	 * @param info - the requested structural info. Can be numSiblings, siblingPosition,
	 * numChildren
	*/
	public String retrieveBrowseStructure(String collection, String service, 
					      String[] classifierIDs, String[] structure, String[] info);

	/**	@return a String representing Greenstone3 
	 * ClassifierBrowseMetadataRetrieve XML giving all the metadata for
	 * all the subclassifiers denoted by nodeIDs. 
	 * @param nodeIDs is of the form CL#.# where the number (#) marks
	 * out structured sections like CL2.1.3. NodeIDs are generally subsections
	 * of top-level classifierNodes (CL#, e.g. CL3).
	 * @param metafields are the classifier metadata fields that are to be returned.
	 * @param collection is the name of the collection 
	 * @param service is the name of the Browse's MetadataRetrieve service 
	 * (usually the browse service is ClassifierBrowse, in which case it always
	 * has a retrieve service called ClassifierBrowseMetadataRetrieve) */
	public String retrieveBrowseMetadata(String collection, String service,
					     String[] nodeIDs, String[] metafields);
	
	/**	@return a String representing Greenstone3 XML for a query process
	 * response returning the results for the query denoted by parameter
	 * nameValParamsMap.
	 * @param nameValParamsMap is a Map of name and value pairs for all the
	 * query field data values. The names match the field names that
	 * describeCollectionService() would have returned for the query service. 
	 * @param collection is the name of the collection 
	 * @param service is the name of the query service
	 * This method is only ever called when any of the services in the digital 
	 * library described themselves as type=query. Therefore any digital
	 * libraries that have no query services, can just return emtpy message 
	 * strings (or even "") since this method will never be called on them 
	 * anyway. */
	public String query(String collection, String service, 
			Map nameValParamsMap);
}