/**
 *#########################################################################
 * FedoraServicesAPIA.java - part of the demo-client for Greenstone 3, 
 * of the Greenstone digital library suite from the New Zealand Digital 
 * Library Project at the  * University of Waikato, New Zealand.
 * <BR><BR>
 * Copyright (C) 2008 New Zealand Digital Library Project
 * <BR><BR>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * <BR><BR>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *########################################################################
 */

package org.greenstone.gs3client.dlservices;

import java.util.Map;
import java.io.File;
import org.greenstone.gs3client.data.DocumentNodeData;
import org.greenstone.fedora.services.*;

/** 
 * FedoraServicesAPIA implements DigitalLibraryServicesAPIA for enabling the 
 * Java-client to access a Fedora repository of Greenstone digital objects. 
 * This is made possible by its use of the FedoraGS3.jar file's functionality, 
 * as made available through the FedoraGS3Connection class. 
 * @author ak19 
*/
public class FedoraServicesAPIA extends FedoraGS3Connection 
	implements DigitalLibraryServicesAPIA
{
	/** Default constructor */
	public FedoraServicesAPIA() throws Exception {
		super(DigitalLibraryServicesAPIA.propertiesFile);
	}

	/** Unused constructor for clients: takes a different properties file */
	public FedoraServicesAPIA(File propertiesFile) throws Exception {
		super(propertiesFile);
	}

	/** Unused constructor for clients: takes a different properties file */
	public FedoraServicesAPIA(String protocol, String host, int port, 
			String fedoraServerUsername, String fedoraServerPassword) throws Exception {
		super(protocol, host, port, fedoraServerUsername, fedoraServerPassword);
	}

	/** @return the name of this digital library for displaying in the client */
	public String getDisplayName() {
		return "Fedora";
	}

	/** @return the directory path to the associated files of the given document 
	 * node.  For instance, the base url of a JEditorPane's HTML documents can be 
	 * set to this. */
	public String getAssocFileBaseURL(DocumentNodeData docNode) {
		return super.getAssocFileBaseURL() 
			+ super.getDocPIDFromDocID(docNode.nodeID) + "/";
	}
	
	/** @return Greenstone3 XML describe response message with information
	 * about collections contained and services (and any serviceRacks) supported
	 * by the Digital library. */
	public String describe() {
		return this.getCollectionList();
	}
	
	/**	@return a String representing Greenstone3 DocumentMetadataRetrieve XML 
	 * containing all the metadata of all the documents indicated by docIDs 
	 * @param collection is the name of the collection 
	 * @param docIDs is an array of document identifiers of documents whose  
	 * metadata is requested 
	 * @param metadata is the list of metadata elements requested. If "all" is specified
	 * then all metadata is requested. */
    public String retrieveDocumentMetadata(String collection, String[] docIDs, String[] metadata) {
	return this.getDocumentMetadata(docIDs, metadata);
	}
	
	/**	@return a String representing Greenstone3 DocumentMetadataRetrieve XML 
	 * containing all the metadata of the document denoted by docID 
	 * @param collection is the name of the collection 
	 * @param docID is the document identifier of the document whose metadata is 
	 * requested.
	 * @param metadata is the list of metadata elements requested. If "all" is specified
	 * then all metadata is requested. */
	public String retrieveDocumentMetadata(String collection, String docID, String[] metadata) {
	    return this.getDocumentMetadata(docID, metadata);
	}

	/**	@return a String representing Greenstone3 DocumentMetadataRetrieve XML 
	 * containing only the title metadata of the documents denoted by docIDs 
	 * @param collection is the name of the collection 
	 * @param docIDs is an array of document identifiers of documents whose titles 
	 * are requested */
	public String retrieveTitleMetadata(String collection, String[] docIDs) {
	    return this.getDocumentMetadata(docIDs, new String[] {"Title"});
	}
	
	/**	@return a String representing Greenstone3 DocumentMetadataRetrieve XML 
	 * containing only the title metadata of the document denoted by docID 
	 * @param collection is the name of the collection 
	 * @param docID is the document identifier of the document whose titles is 
	 * requested */
	public String retrieveTitleMetadata(String collection, String docID) {
	    return this.getDocumentMetadata(docID, new String[] {"Title"});
	}

	
	/**	@return a String representing Greenstone3 DocumentContentRetrieve XML 
	 * containing the document contents of the documents indicated by docIDs 
	 * @param collection is the name of the collection 
	 * @param docIDs is an array of document identifiers of documents whose (text)
	 * contents are requested */
	public String retrieveDocumentContent(String collection, String[] docIDs) {
		return this.getContent(docIDs);
	}
	
	/**	@return a String representing Greenstone3 DocumentContentRetrieve XML 
	 * containing the document contents of the document indicated by docID 
	 * @param collection is the name of the collection 
	 * @param docID is the document identifier of the document whose (text)
	 * content is requested */
	public String retrieveDocumentContent(String collection, String docID) {
		return this.getContent(docID);
	}

	/**	@return a String representing Greenstone3 DocumentStructureRetrieve XML
	 * containing the document structure of the documents indicated by docIDs: 
	 * this means all their descendents 
	 * @param collection is the name of the collection 
	 * @param docIDs is an array of document identifiers of documents whose
	 * hierarchical structures are requested */
	public String retrieveDocumentStructure(String collection, String[] docIDs) {
	    return this.getDocumentStructure(docIDs, new String[]{"descendants"}, new String[]{""});
	}
	
	/**	@return a String representing Greenstone3 DocumentStructureRetrieve XML 
	 * containing the document structure of the document indicated by docID: 
	 * this means all its descendents 
	 * @param collection is the name of the collection 
	 * @param docID is the document identifier of the document whose hierarchical
	 * structure is requested*/
	public String retrieveDocumentStructure(String collection, String docID) {
	    return this.getDocumentStructure(docID, new String[]{"descendants"}, new String[]{""});
	}

	// UNUSED by the client, but still a very useful method:
	/**	@return a String representing Greenstone3 DocumentStructureRetrieve XML 
	 * containing a view of the document structure of the documents denoted by 
	 * docs where only the requested documents and their direct children are  
	 * returned. 
	 * @param collection is the name of the collection 
	 * @param docIDs is an array of document identifiers of documents whose
	 * hierarchical structures are requested */
	public String retrieveDocumentChildren(String collection, String[] docIDs) {
	    return this.getDocumentStructure(docIDs, new String[]{"children"}, new String[]{""});
	}
	
	/**	@return a String representing Greenstone3 DocumentStructureRetrieve XML 
	 * containing a view of the document structure of the document denoted by 
	 * docID where only the document and its direct children are returned. 
	 * @param collection is the name of the collection 
	 * @param docID is the document identifier of the document whose hierarchical
	 * structure is requested */
	public String retrieveDocumentChildren(String collection, String docID) {
	    return this.getDocumentStructure(docID, new String[]{"children"}, new String[]{""});
	}
	
    /** @return a String representing Greenstone3 DocumentMetadataRetrieve XML
	 * containing the requested portion of the document structure of the documents 
	 * indicated by docIDs: 
	 * @param collection is the name of the collection. It's already included in the
	 * docID for a Fedora DL.
	 * @param docIDs is an array of document identifiers of documents whose
	 * hierarchical structures are requested 
	 * @param structure - strings specifying the required structure of each document. 
	 * It can be a combination of: ancestors, parent, siblings, children, descendants, entire.
	 * @param info - strings specifying the required structural info of each document. 
	 * It can be any combination of: siblingPosition, numSiblings, numChildren.
	*/
    public String retrieveDocumentStructure(String collection, String[] docIDs, String[] structure, String[] info) {
	return this.getDocumentStructure(docIDs, structure, info);
    }


    	/** @return a String representing Greenstone3 DocumentMetadataRetrieve XML
	 * containing the requested portion of the document structure of the documents 
	 * indicated by docIDs: 
	 * @param collection is the name of the collection. It's already included in the
	 * docID for a Fedora DL.
	 * @param docID is the document identifier of the document whose hierarchical
	 * structure is requested
	 * @param structure - strings specifying the required structure of the document. 
	 * It can be a combination of: ancestors, parent, siblings, children, descendants, entire.
	 * @param info - strings specifying the required structural info of the document. 
	 * It can be any combination of: siblingPosition, numSiblings, numChildren.
	*/
    public String retrieveDocumentStructure(String collection, String docID, String[] structure, String[] info) {
	return this.getDocumentStructure(docID, structure, info);
    }


    /**	@return a String representing Greenstone3 ClassifierBrowse XML 
	 * giving the entire *structure* of the classification denoted by 
	 * classifierID (including the structures of document descendents of 
	 * the classifier). 
	 * @param classifierIDs - each ID is of the form CL# where the number (#)
	 * marks out structured sections like CL1.1.3 or CL2 
	 * @param collection is the name of the collection 
	 * @param service is the name of the browse service (=ClassifierBrowse usually)
	 * @param structure - the requested browse substructure. Can be any combination
	 * of ancestors, parent, siblings, children, descendants.
	 * @param info - the requested structural info. Can be numSiblings, siblingPosition,
	 * numChildren
	*/
	public String retrieveBrowseStructure(String collection, String service, String[] classifierIDs, 
			String[] structure, String[] info) 
	{
		return this.browse(collection, classifierIDs, structure, info);
	}

	/**	@return a String representing Greenstone3 
	 * ClassifierBrowseMetadataRetrieve XML giving all the metadata for
	 * all the subclassifiers denoted by nodeIDs. 
	 * @param nodeIDs is of the form CL#.# where the number (#) marks
	 * out structured sections like CL2.1.3. NodeIDs are generally subsections
	 * of top-level classifierNodes (CL#, e.g. CL3). 
	 * @param metafields are the classifier metadata fields that are to be returned.
	 * @param collection is the name of the collection 
	 * @param service is the name of the Browse's MetadataRetrieve service 
	 * (usually the browse service is ClassifierBrowse, in which case it always
	 * has a retrieve service called ClassifierBrowseMetadataRetrieve) */
	public String retrieveBrowseMetadata(String collection, String service, 
					     String[] nodeIDs, String[] metafields) 
	{
	    return this.browseMetadataRetrieve(nodeIDs, metafields);
	}
    
       
    // FOR NOW, add the new method that converts URLs to document identifiers(PIDs) here,
    // since it is not used by GS3 classes
    public String getDocIDforURL(String url, String collection) {
	return this.getPIDforURL(url, collection);
    }
	
}
