/**
 *#########################################################################
 * GS3WebServicesAPI.java - part of the demo-client for Greenstone 3, 
 * of the Greenstone digital library suite from the New Zealand Digital 
 * Library Project at the  * University of Waikato, New Zealand.
 * <BR><BR>
 * Copyright (C) 2008 New Zealand Digital Library Project
 * <BR><BR>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * <BR><BR>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *########################################################################
 */

package org.greenstone.gs3client.dlservices;

import java.io.IOException;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Vector;
import java.util.Map;
import java.util.HashMap;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.rpc.ServiceException;

import org.apache.axis.client.Call;
import org.apache.axis.client.Service;
import org.apache.axis.encoding.XMLType;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import org.apache.log4j.Logger; //Import log4j classes

/** 
 * GS3WebServicesQBRAPI deals with invoking the Greenstone 3 web services 
 * functionality through use of Apache Axis' Service and Call objects. 
 * Each Greenstone 3 web service operation has an equivalent method here 
 * that invokes it, even if some of these web service operations are never 
 * called by the Java-client and therefore not prescribed by the 
 * DigitalLibraryServicesAPIA interface. 
 * @author ak19
*/
public class GS3WebServicesQBRAPI {
	/** The Logger for this class */
	static Logger LOG = Logger.getLogger(GS3WebServicesQBRAPI.class);
	
	/** The value that the input dialog's field wsdlURL will default
	 * to if there is (no properties file and) no wsdlURL property */
	public static final String defaultWsdlURL =
		"http://localhost:8080/greenstone3/services/QBRSOAPServerlocalsite?wsdl";
	
	// Service and Call objects
	/** Axis Service object for connecting to invoking Greenstone 3's WebServices
	 * using its WSDL file */
	protected Service service; 
	/** Axis Call object for invoking Greenstone 3's WebServices
	 * using its WSDL file */
	protected Call call;
	
	//maybe unnecessary to store these?
	/** port Name of Greenstone 3's WebServices, as given in the WSDL file */
	protected String portName;
	/** namespace of Greenstone 3's WebServices, as given in the WSDL file */
	protected String namespace;
	/** service Name of Greenstone 3's WebServices, as given in the WSDL file */
	protected String serviceName;
	/** Url of the Greenstone 3 WebServices' WSDL  */
	protected String wsdlURLName;
	
	
	/** GS3ServicesAPIA constructor, that given the url to the wsdl file, finds 
	 * either service and port or the service's endpoint of the GS3 Web Services 
	 * and instantiates the associated Service and Call objects. 
	 * @param wsdlURLName - location of the WSDL URL for Greenstone 3's 
	 * web services */
    public GS3WebServicesQBRAPI(String wsdlURLName) 
    	throws ServiceException, MalformedURLException, 
    		ParserConfigurationException, IOException, SAXException
    {	
    	this.wsdlURLName = wsdlURLName;
    	try {
    		// (1) Try assuming that the wsdlFile URL name is of the form
    		// http://<host>:<port>/greenstone3/services/QBRSOAPServerlocalsite?wsdl
        	// (http://localhost:9090/greenstone3/services/QBRSOAPServerlocalsite?wsdl)
            // then the namespace, serviceName and portName are *likely* to be:
            // - namespace: http://localhost:9090/greenstone3/services/QBRSOAPServerlocalsite
            // - serviceName: QBRSOAPServerlocalsiteService
            // - portName: QBRSOAPServerlocalsite
    		this.namespace = wsdlURLName;
        	int index = namespace.indexOf("?wsdl");
        	if(index != -1)
        		namespace = namespace.substring(0, index);
        	//namespace = "http://gsdl3.greenstone.org";
        	
        	this.portName = namespace; 
    		index = portName.lastIndexOf('/');
    		if(index != -1)
    			portName = portName.substring(index+1, portName.length());
    		
    		this.serviceName = portName + "Service";
    		
    		service = new Service(wsdlURLName, new QName(namespace, serviceName));
    		call = (Call)service.createCall(new QName(namespace, portName));
    	
    	} catch(ServiceException e) {
    		// If we're here, then (1) didn't work. Need to do it the long way
    		// (2) Let's read from the wsdl file and try working out the 
    		// <service> and <port> elements. Some of these have their own  
    		// prefixes: <wsdl:service> and <wsdl:port>, so we need to 
    		// take that into account.
    		
    		// Read from the wsdl file at the URL. There's two ways:
    		// EITHER:
    			//HttpURLConnection wsdlCon = (HttpURLConnection) wsdlURL.openConnection();
    			//InputStreamReader reader = new InputStreamReader(wsdlCon.getInputStream());
    		// OR:
        	URL wsdlURL = new URL(wsdlURLName);
    		InputStreamReader reader = new InputStreamReader(wsdlURL.openStream());
    		
    		// Turn the String xml response into a DOM tree:
    		DocumentBuilder builder = 
    				DocumentBuilderFactory.newInstance().newDocumentBuilder();
    		Document doc 
    			= builder.parse(new InputSource(reader));
    		Element documentElement = doc.getDocumentElement();
    		reader.close();
    		    		
    		// local element names (without any namespace prefixes)
			final String SERVICE = "service"; 
			final String PORT = "port";
			
			// Search for <(any namespace prefix:)service> elements:
			// Note that the API method 
    		// Element.getElementsByTagNameNS("*", SERVICE) 
			// does not return anything because our documentElement is not 
    		// created with DOM Level 2 but DOM Level 1. It won't work therefore. 
			// See the API for Element, spefically for method getElementsByTagNameNS.
			Vector v = getElementsByTagNameNS(documentElement, "*", SERVICE);
			if(v.size() == 0) 
				throw new ServiceException(
						"No <(namespace:)"+SERVICE+"> defined in wsdl");
			Element serviceEl = (Element)v.get(0);
						
			// search for <(any namespace prefix:)port> elements:
			v = getElementsByTagNameNS(serviceEl, "*", PORT);
			if(v.size() == 0)
				throw new ServiceException("No <(namespace:)"+PORT
					+"> defined in wsdl for <(namespace:)service>");
			Element portEl = (Element)v.get(0); //otherwise
			
			final String nameAtt = "name";
			final String targetNamespace = "targetNamespace";
			this.serviceName = serviceEl.hasAttribute(nameAtt) ? 
					serviceEl.getAttribute(nameAtt) : "";
			this.portName = portEl.hasAttribute(nameAtt) ? 
					portEl.getAttribute(nameAtt) : "";
    		
			// reset the namespace, we'll manually try to find it
			this.namespace = ""; 
			// generally namespace is an attribute of the root element 
			// of the wsdl doc
    		if(documentElement.hasAttribute(targetNamespace))
    			this.namespace = documentElement.getAttribute(targetNamespace);
    		
    		if(this.namespace.equals("")) { // we'll still keep looking
	    		NodeList nl = documentElement.getChildNodes();
	    		for(int i = 0; i < nl.getLength(); i++) {
	    			Node n = nl.item(i);
	    			if(n.getNodeType() == Node.ELEMENT_NODE) {
	    				Element el = (Element)n;
	    				if(el.hasAttribute(targetNamespace))
	    					this.namespace = el.getAttribute(targetNamespace);
	    			}
	    		}
    		}
    		
    		// If any of targetNamespace, <service> or <port> were not found,
    		// look for endpointLocation and create empty service and call objects
    		if(namespace.equals("") || serviceName.equals("") || portName.equals("")) {
    			LOG.debug("Namespace " + namespace + "; service " 
    				+ serviceName + "; port " + portName);
    			LOG.debug("Create empty service and call");
    			boolean found = false;
    			// Try to locate the <(wsdl)soap:address> tag.
        		final String SOAP_ADDRESS = "soap:address";
        		final String LOCATION = "location";
    			String endpointLocation = "";
    			// The following method returns all descendent Elements of document element
    			// where the element's tagname *ends* on SOAP_ADDRESS
    			v = getElementsByTagNameNS(portEl, "*", SOAP_ADDRESS);
    			if(v.size() > 0) { //there should be 1 and only 1 such
    				Element soapAddressEl = (Element)v.get(0);
    				if(soapAddressEl.hasAttribute(LOCATION)) {
    					endpointLocation = soapAddressEl.getAttribute(LOCATION);
    					found = true;
    				}
    			}
    			if(!found) { // haven't found endpointLocation, try brute force
    				// Look through all port's *element* children to find some element 
    				// with attribute "location". That could then be the service endpoint...
    				NodeList nl = portEl.getChildNodes();
    				for(int i = 0; !found && i < nl.getLength(); i++) {
    					Node n = nl.item(i); 
    					if(n.getNodeType() == Node.ELEMENT_NODE) {
    						Element el = (Element)n; // we know it's an element
    						if(el.hasAttribute(LOCATION)) {
    							endpointLocation = el.getAttribute(LOCATION);
    							found = true; // found the service endpoint
    						}
    					}
    				}
    			}
    			if(!found) { // still not found any service endpoint
    				throw new ServiceException(
    					"Unable to find service endpoint address. " +
    					"No <(soap:address) location=\"endpoint\" /> in wsdl file"
    				);
    			}
    			service = new Service();
    			call = (Call)service.createCall();
    			call.setTargetEndpointAddress(endpointLocation);
    		} else { // if we had found namespace, serviceName and portName
    			LOG.debug("Namespace " + namespace + "; service " 
    					+ serviceName + "; port " + portName);
    			LOG.debug("Found service, port and namespace");
    			service = new Service(wsdlURL, new QName(namespace, serviceName));
    			call = (Call)service.createCall(new QName(namespace, portName));
    		}
    	} // end catch stmt, which can itself throw ServiceExceptions
    }
  
    /** Static method that gets all the descendant elements of a portion of XMl  
     * within the same namespace as indicated by parameters namespacePrefix and 
     * localName.
     * @param namespacePrefix - a String for the namespacePrefix to search for.
     * This can be the wildcard * for any/allnamespaces.
     * @param localName - the suffix of the namespaceprefix. For instance, localName
     * "soap" will return elements that have the tag wsdlsoap.
     * @param parentElement - the XML element whose descendants will be returned if
     * their element names conform to the specified namespacePrefix and localName
     * parameters.
     * @return a Vector of all descendant elements whose namespace-qualfied tag
     * names are as specified by the parameters namespacePrefix and localName.
     * Had to implement my own getElementsByTagNameNS() method since the original
	 * Element.getElementsByTagNameNS() does not work for Elements created using
	 * DOM level 1 (such as with document.createElement()). See the API for Element.
	 * This method is slightly more generic, as it returns all descendent elements
	 * in a vector where the element name is prefixed by "namespacePrefix" (which 
	 * need not be a namespace uri at all) and is suffixed by an equally arbitrary
	 * String called localName.
	 * If "*" is passed instead of namespacePrefix, this method will ignore any 
	 * prefixes and check only for element names that end on the suffix localName.
	 * 
	 * Old versions of this method had all the same functionality in one method, 
	 * but it was not so optimised (several if-statement checks would have been executed an unnecessary
	 * number of times. This (current) version is split into 3 methods: this one
	 * and two helper functions that deal with the case where namespacePrefix can
	 * be anything and where it is particularly specified.*/
	protected static Vector getElementsByTagNameNS(Element parentElement, 
			String namespacePrefix, String localName)
	{
		Vector v = new Vector();
		if(namespacePrefix.equals("*"))
			getElementsByTagNameSuffix(parentElement, localName, v);
		else // namespacePrefix is specified as something particular
			getElementsByTagNameNS(parentElement, namespacePrefix, localName, v);
		return v;
	}
	
	/** Recursive method.
	 * At method's end, Vector v will contain those descendent elements of 
	 * parentElement where the element's name is suffixed by/ends with parameter
	 * localName.
	 * Related to local method getElementsByTagNameNS(). Deals with the case
	 * where any and all (namespace)Prefix of an element is accepted, but where
	 * an element name's suffix should match with the given localName. 
	 * @param parentElement us the XML element whose descendants are to be 
	 * retrieved by the given TagNameSuffix.
	 * @param localName is the tagName suffix to retrieve the descendants of 
	 * parentElement by. 
	 * @param v is a Vector of all the previously retrieved elements  whose tag
	 * names are suffixed by localName. This Vector is appended to by recursive 
	 * calls to this method. */
	protected static void getElementsByTagNameSuffix(Element parentElement, 
			String localName, Vector v) 
	{
		NodeList allTags = parentElement.getChildNodes();
		for(int i = 0; i < allTags.getLength(); i++) {
			Node n = allTags.item(i);
			if(n.getNodeType() == Node.ELEMENT_NODE) {
				Element e = (Element)n;
				if(e.getNodeName().endsWith(localName))
					v.add(e);
				if(e.hasChildNodes()) // recursive call on all children
					getElementsByTagNameSuffix(e, localName, v);
			}
		}
	}
	
	/** At method's end, Vector v will contain those descendent elements of 
	 * parentElement where the element's name is prefixed by namespacePrefix 
	 * and suffixed by localName. 
	 * Related to local method getElementsByTagNameNS(). Deals with the case
	 * where a particular (namespace)Prefix of an element and suffix (localname)
	 * are given and an element's name should match both prefix and suffix. 
	 * @param parentElement is the XML element whose descendants are to be
	 * retrieved by the given TagNameSuffix.
	 * @param namespacePrefix is the namespace prefix to look for in the 
	 * descendants of parentElement that will be collected in Vector v.
	 * @param localName is the tagName suffix to look for in the descendants of 
	 * parentElement that will be collected in Vector v. 
	 * @param v is a Vector of all the previously retrieved elements whose tag
	 * names are prefixed by namespacePrefix and suffixed by localName. 
	 * This Vector is appended to by recursive calls to this method. */
	protected static void getElementsByTagNameNS(Element parentElement, 
			String namespacePrefix, String localName, Vector v) 
	{
		NodeList allTags = parentElement.getChildNodes();
		for(int i = 0; i < allTags.getLength(); i++) {
			Node n = allTags.item(i);
			if(n.getNodeType() == Node.ELEMENT_NODE) {
				Element e = (Element)n;
				if(e.getNodeName().endsWith(localName) 
						&& e.getNodeName().startsWith(namespacePrefix))
					v.add(e);
				if(e.hasChildNodes()) // recursive call on all children
					getElementsByTagNameNS(e, namespacePrefix, localName, v);
			}
		}
	}
	
    /** @return the namespace of the wsdl file. */
    public String getNamespace() { return namespace; }
    /** @return the web service's serviceName without the namespace */
    public String getService() { return serviceName; }
    /** @return the web service's portName without the namespace */
    public String getPort() { return portName; }
    /** @return the the URL of the wsdl file. */
    public String getWsdl() { return wsdlURLName; }
   	
	//*****************GS3 WEB SERVICE INVOCATION RELATED******************//
    /** Helper method. Uses the Call object to invoke the operation (already
     * set prior to calling invokeWith()) with the given parameters. */
	protected String invokeWith(Object[] params) {
		// the GS3 web service methods all return a String:
		call.setReturnType( XMLType.XSD_STRING );
		try {
			// now call the web service and pass it the parameters
			String response = (String) call.invoke(params);
			return response;
		} catch(Exception e) {
			return "ERROR trying to invoke web service - in " 
				+ this.getClass().getName() 
				+ " invokeWith(Object[] params):\n" + e;
		} finally { // always executed (before return statement/upon completion,
				// regardless of whether an exception occurs or not:
			
			// after invoking, remove all params so that next call to a web 
			// service starts at initial state (no parameters from last time)
			call.removeAllParameters(); 
		}
	}
	
	/** Helper method. Adds a string parameter by the name of paramName to
	 * the Call object. 
	 * @param paramName - the name of the String parameter to add to the call 
	 * object */
	protected void addStringParam(String paramName) {
		 call.addParameter( paramName, 
				 XMLType.XSD_STRING, javax.xml.rpc.ParameterMode.IN );
	}
	
	/** Helper method. Adds an array parameter by the name of arrayParamName 
	 * to the Call object. 
	 * @param arrayParamName - the name of the Array parameter to add to the  
	 * Call object */
	protected void addArrayParam(String arrayParamName) {
		 call.addParameter( arrayParamName, 
				 XMLType.SOAP_ARRAY, javax.xml.rpc.ParameterMode.IN );
	}
	
	/* (1) DESCRIBE MESSAGES, manual pages 35-41 */
	/** Describe request message sent to the Message Router.
	 * @see <a href="http://wiki.greenstone.org/wiki/index.php/Greenstone3">The Greenstone 3 Developer's Manual - pages 35-41</a> 
	 * @param lang is the language of the display content in the response
	 * @param subsetOption is the requested list of items to return in the response
	 * For the Message Router this can be collectionList, serviceClusterList, 
	 * serviceList, siteList */
	public String describe(String lang, String subsetOption) {
		call.setOperationName( "describe" ); //method name is "describe"
		
		addStringParam("lang"); //first param called "lang"
	    addStringParam("subsetOption"); //second param called "subsetOption"
	    
	    return invokeWith(new Object[] {lang, subsetOption});
	}
	
	/** For sending Describe messages to ServiceClusters.
	 * @param serviceCluster is the name of the Service Cluster that this describe 
	 * request is sent to. 
	 * @param lang is the language of the display content in the response
	 * @param subsetOption is the requested list of items to return in the response
	 * For Service Clusters this can be metadataList, serviceList, displayItemList.
	 * @see <a href="http://wiki.greenstone.org/wiki/index.php/Greenstone3">The Greenstone 3 Developer's Manual - pages 35-41</a>
	 */
	public String describeServiceCluster(String serviceCluster, 
			String lang, String subsetOption) 
	{
		call.setOperationName( "describeServiceCluster" );
		
		addStringParam("serviceCluster"); 
		addStringParam("lang");
	    addStringParam("subsetOption"); 
	    
	    return invokeWith(new Object[] { serviceCluster, lang, subsetOption });
	}
	
	/** For sending Describe messages to Collections.
	 * @param collection is the name of the Collection that this describe request 
	 * is sent to. 
	 * @param lang is the language of the display content in the response
	 * @param subsetOption is the requested list of items to return in the response
	 * For Collections this can be metadataList, serviceList and displayItemList.
	 * @see <a href="http://wiki.greenstone.org/wiki/index.php/Greenstone3">The Greenstone 3 Developer's Manual - pages 35-41</a>
	 */
	public String describeCollection(
			String collection, String lang, String subsetOption) 
	{
		call.setOperationName( "describeCollection" );
		
		addStringParam("collection"); 
		addStringParam("lang");
	    addStringParam("subsetOption"); 
	    
	    return invokeWith(new Object[] { collection, lang, subsetOption });
	}
		
	/**
	 * For sending a describe message to a Collection's Service.
	 * @see <a href="http://wiki.greenstone.org/wiki/index.php/Greenstone3">The Greenstone 3 Developer's Manual - pages 35-41</a>  
	 * @param collection is the name of the Collection whose service 
	 * this describe request is sent to.
	 * @param service is the name of the Service (of that collection) to
	 * which this describe request is sent.
	 * @param lang is the language of the display content in the response
	 * @param subsetOption is the requested list of items to return in the response
	 * For Services this can be paramList, displayItemList */
	public String describeCollectionService(String collection, String service, 
			String lang, String subsetOption) 
	{
		call.setOperationName( "describeCollectionService" );
		
		addStringParam("collection");
		addStringParam("service");
		addStringParam("lang");
	    addStringParam("subsetOption"); 
	    
	    return invokeWith(new Object[] {collection, service, lang, subsetOption});
	}
	
	/** 
	 * For sending a describe message to a Service hosted by the Message Router 
	 * (no collection).
	 * @see <a href="http://wiki.greenstone.org/wiki/index.php/Greenstone3">The Greenstone 3 Developer's Manual - pages 35-41</a> 
	 * @param service is the name of the MessageRouter's Service to which this 
	 * describe request is sent.
	 * @param lang is the language of the display content in the response
	 * @param subsetOption is the requested list of items to return in the response
	 * For Services this can be paramList, displayItemList */
	public String describeService(String service, String lang, String subsetOption)
	{
		call.setOperationName( "describeService" );
		
		addStringParam("service");
		addStringParam("lang");
	    addStringParam("subsetOption"); 
	    
	    return invokeWith(new Object[] {service, lang, subsetOption});
	} 
	
	/* (2) Process-type message, QUERY-TYPE SERVICES - p.45 */
	/** For executing a (process-type message) query-type service.
	 * @see <a href="http://wiki.greenstone.org/wiki/index.php/Greenstone3">The Greenstone 3 Developer's Manual - page 45</a>
	 * @param collection is the name of the Collection whose query service this  
	 * query-process request is sent to. If "", then the Message Router is assumed.
	 * @param service is the name of the Query Service (of that collection) to  
	 * which  this request is sent.
	 * @param lang is the language of the display content in the response
	 * @param nameToValsMap is a Map of the (fieldname, value) pairs for the 
	 * parameters of the query. The field names should be those recognised by 
	 * Greenstone 3. That is, the names must exist for the (Collection-)Service Query that this 
	 * message is sent To (as given in 'to' argument). 
	 * For names of arguments, 
	 * @see <a href="http://wiki.greenstone.org/wiki/index.php/Actions_and_Arguments">Greenstone wiki - Actions and Arguments</a>
	 * @see <a href="http://mail-archives.apache.org/mod_mbox/ws-axis-user/200204.mbox/%3C8292F8C4CF63574395CE71CADF765EC30CF46D@neuman.espial.com%3E">How to use XMLType.SOAP_MAP</a>
	*/
	public String query(String collection, String service, 
			String lang, Map nameToValsMap)
	{		
		call.setOperationName( "query" );
		addStringParam("collection");
		addStringParam("service");
		addStringParam("lang");
		call.addParameter("nameToValsMap", XMLType.SOAP_MAP, 
			java.util.Map.class, javax.xml.rpc.ParameterMode.IN);
		return invokeWith(new Object[]{collection, service, lang, nameToValsMap});
	}	
	
        /** 
	 * This method is used to perform the most basic query:
	 * it assumes defaults for all other parameters and provides only
	 * the query string. It is built on top of a TextQuery.
	 * @param collection is the Greenstone collection to be searched
	 * @param lang is the preferred language of the display content in 
	 * the response to be returned.
	 * @param query is the string to be sought in the Greenstone collection
	 * @return a Greenstone 3 XML response message for the query specifying 
	 * the search results.
	 */
	public String basicQuery(String collection, String lang, String query) {
		call.setOperationName( "basicQuery" );
		addStringParam("collection");
		addStringParam("lang");
		addStringParam("query");
		return invokeWith(new Object[]{collection, lang, query});
	}

	/* (3) RETRIEVE PROCESS METHODS - pp.47-49 */
	/** (a) DocumentContentRetrieve request sent to a collection's 
	 * DocumentContentRetrieve service (p.48)
	 * @see <a href="http://wiki.greenstone.org/wiki/index.php/Greenstone3">The Greenstone 3 Developer's Manual - page 48</a>
	 * @param collection is the name of the Collection whose 
	 * DocumentContentRetrieve is requested 
	 * @param lang is the language of the display content in the response 
	 * @param docNodeIDs is the list of documentNodeIDs for which the
	 * content ought to be retrieved.
	*/
	public String retrieveDocumentContent(
			String collection, String lang, String[] docNodeIDs) 
	{
		call.setOperationName( "retrieveDocumentContent" );
		
		addStringParam("collection");
		addStringParam("lang");
		addArrayParam("docNodeIDs");
	    
	    return invokeWith(new Object[] {collection, lang, docNodeIDs});
	}
	
	/** (b) DocumentStructureRetrieve request sent to a collection's 
	 * DocumentStructureRetrieve service (manual pp.48, 49) to retrieve
	 * the entire document structure.
	 * @see <a href="http://wiki.greenstone.org/wiki/index.php/Greenstone3">The Greenstone 3 Developer's Manual - pages 48, 49</a>
	 * @param collection is the name of the Collection whose 
	 * DocumentStructureRetrieve is requested 
	 * @param lang is the language of the display content in the response 
	 * @param docNodeIDs is the list of documentNodeIDs for which the
	 * entire structure ought to be retrieved.
	*/
	public String retrieveEntireDocumentStructure(String collection, 
			String lang, String[] docNodeIDs) 
	{
		call.setOperationName( "retrieveEntireDocumentStructure" );
		
		addStringParam("collection");
		addStringParam("lang");
		addArrayParam("docNodeIDs");
	    
	    return invokeWith(new Object[] {collection, lang, docNodeIDs});
	}
	
	/** DocumentStructureRetrieve request sent to a collection's 
	 * DocumentStructureRetrieve service (manual pp.48, 49) to retrieve
	 * the specified part of the document's structure.
	 * @see <a href="http://wiki.greenstone.org/wiki/index.php/Greenstone3">The Greenstone 3 Developer's Manual - pages 48, 49</a>
	 * @param collection is the name of the Collection whose 
	 * DocumentStructureRetrieve is requested.
	 * @param lang is the language of the display content in the response.
	 * @param docNodeIDs is the list of documentNodeIDs for which the
	 * structure ought to be retrieved. 
	 * @param structure specifies what structure information needs to
	 * be retrieved. The values can be one or more of ancestors, parent, 
	 * siblings, children, descendants (NOTE SPELLING), entire.
	 * @param info - for specifying extra information to be retrieved. 
	 * Possible values for info parameters are numSiblings, siblingPosition, 
	 * numChildren.
	*/
	public String retrieveDocumentStructure(String collection, String lang,
			String[] docNodeIDs, String[] structure, String[] info)
	{
		call.setOperationName( "retrieveDocumentStructure" );
		
		addStringParam("collection");
		addStringParam("lang");
		addArrayParam("docNodeIDs");
		addArrayParam("structure");
		addArrayParam("info");
	    
	    return invokeWith(new Object[] {
	    		collection, lang, docNodeIDs, structure, info});
	}
	
	/** (c) DocumentMetadataRetrieve request sent to a collection's 
	 * DocumentMetadataRetrieve service to retrieve all of a document's metadata. 
	 * @see <a href="http://wiki.greenstone.org/wiki/index.php/Greenstone3">The Greenstone 3 Developer's Manual - page 47</a>
	 * @param collection is the name of the Collection whose 
	 * DocumentMetadataRetrieve is requested.
	 * @param lang is the language of the display content in the response.
	 * @param docNodeIDs is the list of documentNodeIDs for which the
	 * structure ought to be retrieved. 
	*/
	public String retrieveAllDocumentMetadata(String collection, String lang,
			String[] docNodeIDs) 
	{
		call.setOperationName( "retrieveAllDocumentMetadata" );
		
		addStringParam("collection");
		addStringParam("lang");
		addArrayParam("docNodeIDs");
	    
	    return invokeWith(new Object[] {collection, lang, docNodeIDs});
	}
	
	/** DocumentMetadataRetrieve service to retrieve some specific 
	 * metadata values of a document. (Manual on page 47.)
	 * @see <a href="http://wiki.greenstone.org/wiki/index.php/Greenstone3">The Greenstone 3 Developer's Manual - page 47</a>
	 * @param collection is the name of the Collection whose 
	 * DocumentContentRetrieve is requested. 
	 * @param lang is the language of the display content in the response 
	 * @param docNodeIDs is the list of documentNodeIDs for which the
	 * structure ought to be retrieved.
	 * @param metaNames is a list of metadata names which are requested  
	 * to be fetched for the specified documents. 
	*/
	public String retrieveDocumentMetadata(String collection, String lang,
			String[] docNodeIDs, String[] metaNames)
	{
		call.setOperationName( "retrieveDocumentMetadata" );
		
		addStringParam("collection");
		addStringParam("lang");
		addArrayParam("docNodeIDs");
		addArrayParam("metaNames");
	    
	    return invokeWith(new Object[] {collection, lang, docNodeIDs, metaNames});
	}
	
	/** Retrieve all classification Metadata for browsing (sent to the 
	 * ClassifierBrowseMetadataRetrieve service).
	 * @see <a href="http://wiki.greenstone.org/wiki/index.php/Greenstone3">The Greenstone 3 Developer's Manual - pages 47, 48</a>
	 * @param collection is the name of the Collection whose 
	 * ClassifierBrowseMetadataRetrieve service is called 
	 * @param categoryName - name of the browsing category, usually
	 * ClassifierBrowse. (If left as "", then it defaults to ClassifierBrowse) 
	 * @param lang is the language of the display content in the response 
	 * @param nodeIDs is the list of document or classifier NodeIDs 
	 * for which the metadata ought to be retrieved.
	*/
	public String retrieveAllBrowseMetadata(String collection, 
			String categoryName, String lang, String[] nodeIDs) 
	{
		call.setOperationName( "retrieveAllBrowseMetadata" );
		
		addStringParam("collection");
		addStringParam("categoryName");
		addStringParam("lang");
		addArrayParam("nodeIDs");
	    
	    return invokeWith(new Object[] {collection, categoryName, lang, nodeIDs});
	}
	
	/**  ClassifierBrowseMetadataRetrieve service to retrieve some specific 
	 * metadata values of a document. 
	 * @see <a href="http://wiki.greenstone.org/wiki/index.php/Greenstone3">The Greenstone 3 Developer's Manual - pages 47, 48</a> 
	 * @param collection is the name of the Collection whose 
	 * ClassifierBrowseMetadataRetrieve service is called
	 * @param categoryName - name of the browsing category, usually
	 * "ClassifierBrowse". (If left as "", then it defaults to ClassifierBrowse) 
	 * @param lang is the language of the display content in the response 
	 * @param nodeIDs is the list of document or classifier NodeIDs 
	 * for which the metadata ought to be retrieved.
	 * @param metaNames is a list of metadata names which are requested  
	 * to be fetched for the specified documents or classifiers */
	public String retrieveBrowseMetadata(String collection, String categoryName,  
			String lang, String[] nodeIDs, String[] metaNames) 
	{
		call.setOperationName( "retrieveBrowseMetadata" );
		
		addStringParam("collection");
		addStringParam("categoryName");
		addStringParam("lang");
		addArrayParam("nodeIDs");
		addArrayParam("metaNames");
		
	    return invokeWith(new Object[] {collection, categoryName, lang, 
	    		nodeIDs, metaNames});
	}
	
	/* (4) Classifier BROWSE PROCESS METHODS */
	/** To send a browse request for all the descendants of a classifier node.
	 * Useful for getting the entire structure of a top-level 
	 * &lt;classificationNode&gt;.
	 * @see <a href="http://wiki.greenstone.org/wiki/index.php/Greenstone3">The Greenstone 3 Developer's Manual - page 46</a>
	 * @param collection is the name of the Collection whose browse Classifier
	 * Browse Service is called
	 * @param browseService is the name of the (Classifier) Browse Service (of 
	 * the given collection) to which this request message is sent. 
	 * @param lang is the language of the display content in the response 
	 * @param classifierNodeIDs is an array of classifierNodeIDs for which the
	 * structures ought to be retrieved.
	*/
	public String browseDescendants(String collection, String browseService, 
			String lang, String[] classifierNodeIDs) 
	{
		call.setOperationName( "browseDescendants" );
		
		addStringParam("collection");
		addStringParam("browseService");
		addStringParam("lang");
		addArrayParam("classifierNodeIDs");
	    
	    return invokeWith(new Object[] {
	    		collection, browseService, lang, classifierNodeIDs});
	}
	
	/** To send a browse request for specific parts of a classifier node 
	 * (children, ancestors, descendants). Useful for getting specific parts
	 * of the structure of a top-level &lt;classificationNode&gt;.
	 * @see <a href="http://wiki.greenstone.org/wiki/index.php/Greenstone3">The Greenstone 3 Developer's Manual - page 46</a>
	 * @param collection is the name of the Collection whose browse Classifier
	 * Browse Service is called
	 * @param browseService is the name of the (Classifier) Browse Service (of 
	 * the given collection) to which this request message is sent. 
	 * @param lang is the language of the display content in the response 
	 * @param classifierNodeIDs is the list of classifierNodeIDs for which the
	 * structure ought to be retrieved.
	 * @param structureParams the list of parameters indicating what structure 
	 * information is requested. Accepted values are ancestors, parent, siblings,
	 * children, descendants.
	 * @param infoParams - the requested structural info. Can be numSiblings, siblingPosition,
	 * numChildren
	*/
	public String browse(String collection, String browseService, String lang, 
			     String[] classifierNodeIDs, String[] structureParams, String[] infoParams)
	{
		call.setOperationName( "browse" );
		
		addStringParam("collection");
		addStringParam("browseService");
		addStringParam("lang");
		addArrayParam("classifierNodeIDs");
	    addArrayParam("structureParams");
	    addArrayParam("infoParams");
	    
	    return invokeWith(new Object[] {collection, browseService, lang, 
					    classifierNodeIDs, structureParams, infoParams});
	}
		
	/** @return a help string for listing all the web service methods. */
	public String help() {
		call.setOperationName( "help" );
		return invokeWith(new Object[] {});
	}
	
	/** @param methodname is the name of the method to be described.  
	 * @return a help string for the given method, explaining what the method
	 * does, what parameters it expects and their types and what it returns.
	*/
	public String helpWithMethod(String methodname) {
		call.setOperationName( "helpWithMethod" );
		addStringParam("methodname");
		return invokeWith(new Object[] {methodname});
	}
	
	public static void main(String args[]) {
		try{
			GS3WebServicesQBRAPI ws = new GS3WebServicesQBRAPI(GS3ServicesAPIA.showWsdlInputDialog());
			
			// (1) trying 4 describe methods
			System.out.println(ws.describe("", ""));
			System.out.println(ws.describeCollection("gs2mgppdemo", "", ""));
			// ? //System.out.println(ws.describeServiceCluster("gs2mgppdemo", "", ""));
			System.out.println(ws.describeCollectionService("gs2mgppdemo", "AdvancedFieldQuery", "", ""));
			System.out.println(ws.describeService("AddItem", "", ""));
			
			// (2) try 2 query methods: query and basicQuery
			HashMap map = new HashMap();
			map.put("maxDocs", "100");
			map.put("level", "Sec");
			map.put("accent", "1");
			map.put("matchMode", "some");
			map.put("fqf", "ZZ,ZZ,ZZ,ZZ");
			map.put("case", "1");
			map.put("sortBy", "1");
			map.put("fqv", "snail,water,,");
			
			String response = ws.query("gs2mgppdemo", "FieldQuery", "", map);
			System.out.println("Regular query: " + response);
			
			// basicQuery
			response = ws.basicQuery("gs2mgppdemo", "", "cat");
			System.out.println("Basic query on 'cat' over collection gs2mgppdemo:"
					+ response);
			
			// (3) try 2 browse
			System.out.println("describe browse:\n" 
					+ ws.describeCollectionService("gs2mgppdemo", "ClassifierBrowse", "", ""));

			System.out.println("browse children of CL1-CL4:\n" +
					ws.browse("gs2mgppdemo", "ClassifierBrowse", "", 
					new String[]{"CL1", "CL2", "CL3", "CL4"}, new String[]{"children"}, new String[]{"numChildren"}));
			
			System.out.println("browse descendants of CL2.3:\n" +
					ws.browseDescendants("gs2mgppdemo", "ClassifierBrowse", "", 
					new String[]{"CL2.3"}));
			
			// (4) try 2 DocStructure
			System.out.println("retrieve ancestors and children structure of HASH016193b2847874f3c956d22e.4:\n" +
					ws.retrieveDocumentStructure("gs2mgppdemo", "", 
							new String[]{"HASH016193b2847874f3c956d22e.4"}, 
							new String[]{"ancestors", "children"}, new String[]{"numSiblings"}));
							
			
			System.out.println("retrieve entire structure of HASH016193b2847874f3c956d22e.4.1:\n" +
					ws.retrieveEntireDocumentStructure(
							"gs2mgppdemo", "", new String[]{"HASH016193b2847874f3c956d22e.4.1"}));
			
			
			// (5) try the 1 DocumentContent retrieve
			System.out.println("retrieve content of HASH016193b2847874f3c956d22e.4:\n" +
					ws.retrieveDocumentContent("gs2mgppdemo", "", 
							new String[]{"HASH016193b2847874f3c956d22e.4", "HASH016193b2847874f3c956d22e.4.1", 
							"HASH016193b2847874f3c956d22e.4.2"}));
			
			
			// (6) try 2 DocumentMeta
			System.out.println("retrieve title meta of HASH016193b2847874f3c956d22e.4 children:\n" +
					ws.retrieveDocumentMetadata("gs2mgppdemo", "", 
					new String[]{"HASH016193b2847874f3c956d22e.4.1", "HASH016193b2847874f3c956d22e.4.2",
					"HASH016193b2847874f3c956d22e.4.3"}, new String[]{"Title"}));
			
			System.out.println("retrieve all meta of HASH016193b2847874f3c956d22e.4 children:\n" +
					ws.retrieveAllDocumentMetadata("gs2mgppdemo", "", 
							new String[]{"HASH016193b2847874f3c956d22e.4",
							"HASH016193b2847874f3c956d22e.4.1", "HASH016193b2847874f3c956d22e.4.2",
						"HASH016193b2847874f3c956d22e.4.3"}));
			
			// (7) try 2 BrowseMeta
			System.out.println("retrieve all browse meta of CL1, CL2, CL2.1, CL3:\n" +
					ws.retrieveAllBrowseMetadata("gs2mgppdemo", "", "", 
								     new String[]{"CL1", "CL2", "CL2.1", "CL3"}));
			
			System.out.println("retrieve Title, hastxt browse meta of CL1, CL2, CL2.1, CL3:\n" +
					ws.retrieveBrowseMetadata("gs2mgppdemo", "", "", 
					new String[]{"CL1", "CL2","CL2.1", "CL3"}, new String[]{"Title", "hastxt"}));
			
			// (8) try help
			System.out.println(ws.help());
			System.out.println(ws.helpWithMethod("describe"));
			
			/*
			// need to make <something>SOAPServer's "properties" member variable public 
			// before these can be tested:
			System.out.println("Calling webservices HELP:\n" + org.greenstone.gs3services.QBRSOAPServer.help());
			java.util.Enumeration props = org.greenstone.gs3services.QBRSOAPServer.properties.keys();
			while(props.hasMoreElements()){
				String methodName = (String)props.nextElement();
				System.out.println(//"method: " + methodName + "\n" + 
					org.greenstone.gs3services.QBRSOAPServer.helpWithMethod(methodName));
			}
			System.out.println("Calling webservices HELP:\n" + org.greenstone.gs3services.AdminSOAPServer.help());
			java.util.Enumeration props = org.greenstone.gs3services.AdminSOAPServer.properties.keys();
			while(props.hasMoreElements()){
				String methodName = (String)props.nextElement();
				System.out.println(//"method: " + methodName + "\n" + 
					org.greenstone.gs3services.AdminSOAPServer.helpWithMethod(methodName));		
			}*/
				
		}catch(Exception e) {
			System.out.println("Problem accessing service\n" + e.getMessage());
			e.printStackTrace();
		}
	}
}