/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Copyright (C) 2022 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.cdm;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;

import org.greenstone.gatherer.Dictionary;
import org.greenstone.gatherer.util.StaticStrings;
import org.w3c.dom.*;

/** handles sortfields for lucene and solr collections */
public class SortFieldManager extends BaseIndexManager {

    static final private String NONE = "none";
    static final private String RANK = "rank";

    public SortFieldManager(Element sortfields, String current_build_type) {
	super(sortfields, current_build_type, StaticStrings.SORT_ELEMENT, StaticStrings.SORT_DEFAULT_ELEMENT, new SortField());
	this.controls_title_key = "CDM.SortFieldManager.Title";
	this.new_button_tooltip_key = "CDM.SortFieldManager.New_Button_Tooltip";
	this.edit_button_tooltip_key = "CDM.SortFieldManager.Edit_Button_Tooltip";
	this.remove_button_tooltip_key = "CDM.SortFieldManager.Remove_Button_Tooltip";
        this.default_indicator_key = "CDM.SortFieldManager.Default_Sort_Indicator";
        this.set_default_tooltip_key = "CDM.SortFieldManager.Set_Default_Tooltip";
	this.nip_new_index_key = "CDM.SortFieldManager.New";
	this.nip_edit_index_key = "CDM.SortFieldManager.Edit";
	this.nip_source_label_key = "CDM.SortFieldManager.Source";
        this.nip_source_tooltip_key = "CDM.SortFieldManager.Source_Tooltip";
	this.nip_custom_label_key = "CDM.IndexManager.Custom";
	this.nip_custom_tooltip_key = "CDM.IndexManager.Custom_Tooltip";

	this.nip_add_index_button_key = "CDM.SortFieldManager.Add";
	this.nip_add_index_tooltip_key = "CDM.SortFieldManager.Add_Tooltip";
	this.nip_replace_index_button_key = "CDM.SortFieldManager.Replace";
	this.nip_replace_index_tooltip_key = "CDM.SortFieldManager.Replace_Tooltip";
	this.nip_add_all_index_button_key = "CDM.IndexManager.AddAll";
      this.nip_add_all_index_tooltip_key = "CDM.SortFieldManager.AddAll_Tooltip";

    }

    public void buildTypeChanged(String new_build_type) {
	if (new_build_type.equals(BuildTypeManager.BUILD_TYPE_SOLR)|| new_build_type.equals(BuildTypeManager.BUILD_TYPE_LUCENE)) {
	    // activate
	    setAssigned(true);
	} else {
	    // deactivate
	    setAssigned(false);
	}
    }

    public Control getControls() {
	if (controls == null) {
	    controls = new SortFieldControl();
	}
	return controls;
    }

    private class SortFieldControl
	extends IndexControl {

	public SortFieldControl() {
	    super();	    
	}

	/** we want our own custom new index prompt for searhc indexes */
	protected NewIndexPrompt createNewIndexPrompt(String build_type, Index index) {
	    return new NewSortFieldPrompt(build_type, index);
	    
	}

	// we extend this to add in the rank/none options
	protected class NewSortFieldPrompt
	    extends NewIndexPrompt {

	    private JCheckBox none_checkbox;
	    private JCheckBox rank_checkbox;

          public NewSortFieldPrompt(String build_type, Index existing_index) {
            super(build_type, existing_index);	    
          }
          
          /** inside here is where we customise our controls */
          protected void generateContents(String build_type, Index existing_index) {
	      super.generateContents(build_type, existing_index);
	      none_checkbox = new JCheckBox(Dictionary.get("CDM.SortFieldManager.Field_None"));
              none_checkbox.addItemListener(new NoneRankBoxListener());
              none_checkbox.setComponentOrientation(Dictionary.getOrientation());
	      none_checkbox.setToolTipText(Dictionary.get("CDM.SortFieldManager.Field_None_Tooltip"));

	      rank_checkbox = new JCheckBox(Dictionary.get("CDM.SortFieldManager.Field_Rank"));
              rank_checkbox.addItemListener(new NoneRankBoxListener());
              rank_checkbox.setComponentOrientation(Dictionary.getOrientation());
	      rank_checkbox.setToolTipText(Dictionary.get("CDM.SortFieldManager.Field_Rank_Tooltip"));

	      // fill in none/rank if it is selected in existing index
	      if (existing_index !=null) {
		  ArrayList sources = existing_index.getSources();
		  if (sources.contains(NONE)) {
		      none_checkbox.setSelected(true);
		      rank_checkbox.setEnabled(false);
		      source_list.setEnabled(false);
		      custom_field.setEnabled(false);
		  } else if (sources.contains(RANK)) {
		      rank_checkbox.setSelected(true);
		      none_checkbox.setEnabled(false);
		      source_list.setEnabled(false);
		      custom_field.setEnabled(false);
		  }
		  
	      }

	      extra_pane.add(none_checkbox);
	      extra_pane.add(rank_checkbox);

	  }
	    // override this to take into account RANK/NONE options
	    protected String getCustomItemsFromSources(ArrayList sources) {
		ArrayList all_items = source_list.getAll();
		String result = "";
		for (int i=0; i < sources.size(); i++) {
		    Object item = sources.get(i);
		    if (!all_items.contains(item)&& !item.equals(RANK) && !item.equals(NONE)) {
			result += item+",";
		    }
		}
		return result;

	    }
	    // override this to take into account none/rank options
	    protected Index generateNewIndex() {
		ArrayList sources;

		String custom_text = custom_field.getText();
		if (none_checkbox.isSelected()) {
		    sources = new ArrayList();
		    sources.add(NONE);
		} else if (rank_checkbox.isSelected()) {
		    sources = new ArrayList();
		    sources.add(RANK);
		} else if (!source_list.isNothingTicked() || !custom_text.equals("")) {
		    sources = source_list.getTicked();
		    addCustomItemsToSources(sources, custom_text);
		} else {
		    // nothing selected
		    return null;
		}
		return new SortField(sources);

	    }


	    private class NoneRankBoxListener
		implements ItemListener {

		public void itemStateChanged(ItemEvent event) {

		    if (none_checkbox.isSelected() ) {
			source_list.setEnabled(false);
			rank_checkbox.setEnabled(false);
			custom_field.setEnabled(false);
		    } else if (rank_checkbox.isSelected()) {
			source_list.setEnabled(false);
			none_checkbox.setEnabled(false);
			custom_field.setEnabled(false);
		    } else {
			// neither button is selected
			none_checkbox.setEnabled(true);
			rank_checkbox.setEnabled(true);
			source_list.setEnabled(true);
			custom_field.setEnabled(true);
		    }
		    validateAddOrReplaceButton();
		}
		
	    }
	}
	    
    }
}
