/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * <BR><BR>
 *
 * Author: John Thompson, Greenstone Digital Library, University of Waikato
 *
 * <BR><BR>
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * <BR><BR>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * <BR><BR>
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * <BR><BR>
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.gui;

import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.filechooser.*;
import javax.swing.table.*;
import org.greenstone.gatherer.Configuration;
import org.greenstone.gatherer.Dictionary;
import org.greenstone.gatherer.Gatherer;
import org.greenstone.gatherer.util.Utility;

/** This dialog allows the user to choose which browser to use for the preview command 
 * @author Katherine Don, Greenstone Digital Library, University of Waikato
 * @version 2.3
 */
public class PreviewCommandDialog
    extends ModalDialog {

    /** The default size for the dialog. */
    static final private Dimension SIZE = new Dimension(500, 195);

    private PreviewCommandDialog self;
    private JButton browse_button;
    private JButton cancel_button;
    private JButton ok_button;
    private JTextField command_field;
    private String preview_command=null;
    /** Create a new PreviewCommandDialog
     */
    public PreviewCommandDialog() {
	super(Gatherer.g_man);
	this.self = this;
        JScrollPane scrol_tmp;
        this.setComponentOrientation(Dictionary.getOrientation());
	// Creation
	setModal(true);
	setSize(SIZE);
	setJMenuBar(new SimpleMenuBar("thepreviewview"));
	setTitle(Dictionary.get("PreviewCommandDialog.Title"));
	
	JPanel content_pane = (JPanel) getContentPane();
        content_pane.setComponentOrientation(Dictionary.getOrientation());
	content_pane.setBackground(Configuration.getColor("coloring.collection_heading_background", false));

	JTextArea instructions_area = new JTextArea(Dictionary.get("PreviewCommandDialog.Instructions"));
        instructions_area.setComponentOrientation(Dictionary.getOrientation());
        instructions_area.setEditable(false);
	instructions_area.setLineWrap(true);
	instructions_area.setRows(5);
	instructions_area.setWrapStyleWord(true);
	
	JPanel button_pane = new JPanel();
        button_pane.setComponentOrientation(Dictionary.getOrientation());
	JPanel lower_pane = new JPanel();
        lower_pane.setComponentOrientation(Dictionary.getOrientation());

	JPanel command_pane = new JPanel();
        command_pane.setComponentOrientation(Dictionary.getOrientation());
	command_field = new JTextField();
        command_field.setComponentOrientation(Dictionary.getOrientation());
	browse_button = new GLIButton(Dictionary.get("FileAssociationDialog.Browse"));
	browse_button.setEnabled(!Utility.isMac());
	if (Utility.isMac()) {
	    browse_button.setToolTipText(Dictionary.get("FileAssociationDialog.Browse_Tooltip_Mac"));
	} else {
	    browse_button.setToolTipText(Dictionary.get("FileAssociationDialog.Browse", "FileAssociationDialog.Browse_Tooltip"));
	}
	ok_button = new GLIButton(Dictionary.get("General.OK"), Dictionary.get("FileAssociationDialog.Close_Tooltip"));
	cancel_button = new GLIButton(Dictionary.get("General.Cancel"), Dictionary.get("General.Pure_Cancel_Tooltip"));
	
	// Connection
	browse_button.addActionListener(new BrowseButtonListener());
	ok_button.addActionListener(new OkButtonListener());
	cancel_button.addActionListener(new CancelButtonListener());
	
	// Layout
	command_pane.setBorder(BorderFactory.createEmptyBorder(2,0,2,0));
	command_pane.setLayout(new BorderLayout());
	command_pane.add(command_field, BorderLayout.CENTER);
	command_pane.add(browse_button, BorderLayout.LINE_END);

	lower_pane.setBorder(BorderFactory.createEmptyBorder(2,0,0,0));

	button_pane.setLayout(new GridLayout(1,2,5,0));
	button_pane.add(ok_button);
	button_pane.add(cancel_button);

	lower_pane.setBorder(BorderFactory.createEmptyBorder(2,0,0,0));
	lower_pane.setLayout(new BorderLayout());
	lower_pane.add(command_pane, BorderLayout.CENTER);
	lower_pane.add(button_pane, BorderLayout.SOUTH);


	content_pane.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
	content_pane.setLayout(new BorderLayout());
        scrol_tmp=new JScrollPane(instructions_area);
        scrol_tmp.setComponentOrientation(Dictionary.getOrientation());
	content_pane.add(scrol_tmp, BorderLayout.NORTH);
	content_pane.add(lower_pane, BorderLayout.CENTER);

	Rectangle screen = Gatherer.g_man.getBounds(null);
	setLocation((int)(screen.getX() + (screen.getWidth() - SIZE.width) / 2), (int)(screen.getY() + (screen.getHeight() - SIZE.height) / 2));
	screen = null;
    }

    public void destroy() {
	// Disconnect
	// Clean up
	self = null;
    }

    /** Redisplay the dialog
     */
    public String display() {
	setVisible(true);
	return preview_command;
    }


    private class OkButtonListener
	implements ActionListener {
	
	public void actionPerformed(ActionEvent event) {
	    preview_command = command_field.getText();
	    self.dispose();
	    
	}
    }


    /** Whenever the user clicks the browse button, we should open up a file browser to allow them to select an executable file from somewhere in the file system. */
    private class BrowseButtonListener 
	implements ActionListener {
	/** Open up a simple JFileChooser when the user clicks the button. 
	 * @param event An <strong>ActionEvent</strong> containing information about the event.
	 */
	public void actionPerformed(ActionEvent event) {
	    JFileChooser chooser = new JFileChooser(new File(Gatherer.getGLIUserDirectoryPath()));
            chooser.setComponentOrientation(Dictionary.getOrientation());
	    GUIUtils.disableRename(chooser);
	    chooser.setDialogTitle(Dictionary.get("FileAssociationDialog.Browse_Title"));
	    chooser.setFileFilter(new BatchFileFilter());
	    chooser.setFileFilter(new CoreObjectModelFileFilter());
	    chooser.setFileFilter(new ExecutableFileFilter());
	    chooser.setAcceptAllFileFilterUsed(true);
	    if(chooser.showOpenDialog(Gatherer.g_man) == JFileChooser.APPROVE_OPTION) {
		command_field.setText(chooser.getSelectedFile().getAbsolutePath());
	    }
	}
    }

    
    private class CancelButtonListener
	implements ActionListener {
	public void actionPerformed(ActionEvent event) {
	    self.dispose();
	}
    }

    /** Batch filter shows only files ending in bat. Based on ImageFilter.java which is a 1.4 example used by FileChooserDemo2.java. */
    private class BatchFileFilter
	extends FileFilter {

	/** Accept all .exe files
	 * @param file a File
	 * @return true is this file should be shown, false otherwise
	 */
	public boolean accept(File file) {
	    return file.getName().toLowerCase().endsWith(".bat");
	}

	/** The description of this filter
	 * @return a String
	 */
	public String getDescription() {
	    return Dictionary.get("FileAssociationDialog.Batch_File");
	}
    }

    /** Command filter shows only files ending in com. Based on ImageFilter.java which is a 1.4 example used by FileChooserDemo2.java. */
    private class CoreObjectModelFileFilter
	extends FileFilter {

	/** Accept all .com files
	 * @param file a File
	 * @return true is this file should be shown, false otherwise
	 */
	public boolean accept(File file) {
	    return file.getName().toLowerCase().endsWith(".com");
	}

	/** The description of this filter
	 * @return a String
	 */
	public String getDescription() {
	    return Dictionary.get("FileAssociationDialog.Command_File");
	}
    }


    /** Executable filter shows only files ending in exe. Based on ImageFilter.java which is a 1.4 example used by FileChooserDemo2.java. */
    private class ExecutableFileFilter
	extends FileFilter {

	/** Accept all .exe files
	 * @param file a File
	 * @return true is this file should be shown, false otherwise
	 */
	public boolean accept(File file) {
	    return file.getName().toLowerCase().endsWith(".exe");
	}

	/** The description of this filter
	 * @return a String
	 */
	public String getDescription() {
	    return Dictionary.get("FileAssociationDialog.Executable_File");
	}
    }


}






