/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * <BR><BR>
 *
 * Author: John Thompson, Greenstone Digital Library, University of Waikato
 *
 * <BR><BR>
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * <BR><BR>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * <BR><BR>
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * <BR><BR>
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.shell;

import java.io.*;
import java.net.*;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.regex.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;
import org.greenstone.gatherer.Configuration;
import org.greenstone.gatherer.DebugStream;
import org.greenstone.gatherer.Dictionary;
import org.greenstone.gatherer.Gatherer;
import org.greenstone.gatherer.cdm.CollectionConfiguration;
import org.greenstone.gatherer.cdm.CollectionDesignManager;
import org.greenstone.gatherer.cdm.CollectionMetaManager;
import org.greenstone.gatherer.cdm.CollectionMeta;
import org.greenstone.gatherer.collection.CollectionManager;
import org.greenstone.gatherer.metadata.DocXMLFileManager;
import org.greenstone.gatherer.remote.RemoteGreenstoneServer;
import org.greenstone.gatherer.util.SafeProcess;
import org.greenstone.gatherer.util.StaticStrings;
import org.greenstone.gatherer.util.Utility;


/** The <strong>GShell</strong> is reponsible for running a separately threaded process in the command shell. This is necessary for executing the Perl Scripts and also for other system related funcitonality.
 * When modifying this class, bear in mind concurrency issues that could arise with SafeProcess'
 * worker threads and where synchronization may be needed to prevent such issues.
 */
public class GShell
    extends Thread implements SafeProcess.ExceptionHandler {
    /** A flag used to determine if this process has been asked to cancel. */
    private boolean cancel = false;
    private BufferedOutputStream buffered_output_stream = null;
    /** The list of listeners associated with this class. 
     * Make it a CopyOnWriteArrayList to make it threadsafe at a cost of decreased efficiency. 
     * see http://stackoverflow.com/questions/8259479/should-i-synchronize-listener-notifications-or-not
     */
    private final CopyOnWriteArrayList<GShellListener> listeners;
    /** The current status of this shell process. */
    private int status = -1;
    /** The type of message being sent. */
    private int msg_type = -1;
    /** The type of shell process. */
    private int type = -1;
    /** The caller of this process, and thus the class most interested in messages. */
    private GShellListener caller = null;
    /** The progress monitor associated with this process. */
    private GShellProgressMonitor progress = null;
    /** Arguments to be given to the process (including the executable you are calling. */
    private String args[] = null;
    /** The command_output returned from executing a process */
    private String commandOutput = null;

    /** The process safely run in this GShell Thread. Make sure to set to prcs to null when done with. */
    private SafeProcess prcs = null;

    /** Elements in process type enumeration. */
    static final public int BUILD = 0;
    static final public int IMPORT = 1;
    static final public int NEW = 2;
    static final public int EXPORTAS = 3;
    static final public int CDIMAGE = 4;
    static final public int CONVERT = 5;
    static final public int EXPLODE = 6;
    static final public int SRCREPLACE = 7; // for replacing source docs with their html
    static final public int SCHEDULE = 8;
    static final public int DELETE = 9;

    /** Elements in status type enumeration. */
    static final public int ERROR = 0;
    static final public int OK = 1;
    static final public int CANCELLED = 2;

    /** Elements in process type name enumeration. */
    static public String GSHELL_BUILD = "gshell_build";
    static public String GSHELL_IMPORT = "gshell_import";
    static public String GSHELL_NEW = "gshell_new";
    static public String GSHELL_EXPORTAS = "gshell_exportas";
    static public String GSHELL_CDIMAGE = "gshell_cdimage";
    static public String GSHELL_CONVERT = "gshell_convert";
    static public String GSHELL_EXPLODE = "gshell_explode";
    static public String GSHELL_SRCREPLACE = "gshell_srcreplace"; // for replacing source docs with their html versions
    static public String GSHELL_SCHEDULE = "gshell_schedule"; 
    static public String GSHELL_FEDORA_COLDELETE = "gshell_fedora_col_delete";


    /** Constructor gatherer all the data required to create a new process, and emit meaningfull messages.
     * @param args A <strong>String[]</strong> containing the arguments to the process thread, including the name of the executable.
     * @param type An <strong>int</strong> that indicates what group of processes this process belongs to, as some are treated slightly differently (i.e an IMPORT type process is always followed by a BUILD one).
     * @param msg_type As process threads may be background (like a makecol.pl call) or important processes in their own right (such as an IMPORT-BUILD) we must be able to set what level messages posted by this class will have by usings this <strong>int</strong>.
     * @param caller The default <i>GShellListener</i> that is interested in the progress of this process.
     * @param progress The <i>GShellProgressMonitor</i> associated with this process.
     * @param name A <strong>String</strong> identifier given to the process, for convience and debug reasons.
     */
    public GShell(String args[], int type, int msg_type, GShellListener caller, GShellProgressMonitor progress, String name) {
	super(name);
	this.args = args;
	this.msg_type = msg_type;
	this.type = type;
	this.caller = caller;
	this.progress = progress;
	this.status = 0;
	// Lower this jobs priority
	this.setPriority(Thread.MIN_PRIORITY);
	listeners = new CopyOnWriteArrayList<GShellListener>();
	if(caller != null) {
	    listeners.add(caller);
	}
    }
    /** This method adds another shell listener to this process.
     * @param listener The new <i>GShellListener</i>.
     */
    public void addGShellListener(GShellListener listener) {
	listeners.add(listener);
    }
    /** This method removes a certain shell listener from this process.
     * @param listener The <i>GShellListener</i> to be removed.
     */
    /* private void removeGShellListener(GShellListener listener) {
       listeners.remove(listener);
       } */

    // This method is only used by old_runLocal(). If that is removed, then remove this too
    protected StringBuffer get_stream_char(InputStreamReader isr, StringBuffer line_buffer,
				   BufferedOutputStream bos) throws IOException 
    {
	int c = isr.read();
	///atherer.println("isr: '" + (char) c + "'");
	if(c == '\n' || c == '\r') {
	    if(line_buffer.length() > 0) {
		String line = line_buffer.toString();
		// DebugStream.println("* " + line + " *");		
		fireMessage(type, typeAsString(type) + "> " + line, status, bos);
		line_buffer = new StringBuffer();
	    }
	}
	else {
	    line_buffer.append((char)c);
	}

	return line_buffer;
    }

    public String getCommandOutput() { return commandOutput; }
    public void resetCommandOutput() { commandOutput = null; }

    private void runRemote(String[] args, BufferedOutputStream bos)
    {
	// Make sure the process hasn't been cancelled
	if (hasSignalledStop()) {
	    return;
	}

	try {
	    int directory_name_end = args[0].lastIndexOf(File.separator);
	    String script_name = ((directory_name_end != -1) ? args[0].substring(directory_name_end + 1) : args[0]);
	    System.err.println("Script name: " + script_name);
	    String collection_name = args[args.length - 1];
	    System.err.println("Collection name: " + collection_name);


	    String script_args = "";
	    for (int i = 1; i < (args.length - 1); i++) {
		// Skip arguments that won't make sense on the server
		// and -site is added in RemoteGreenstonServer.java
		if (args[i].equals("-collectdir") || args[i].equals("-importdir") || args[i].equals("-builddir") || args[i].equals("-site")) {
		    i++;
		    continue;
		}

		// Script arguments get changed to CGI arguments
		if (args[i].startsWith("-")) {
		    script_args += "&" + args[i].substring(1) + "=";
		    if ((i + 1) < (args.length - 1) && !args[i + 1].startsWith("-")) {
			script_args += URLEncoder.encode(args[i + 1], "UTF-8");
			i++;
		    }
		}
	    }

	    System.err.println("Script args: " + script_args);
	    buffered_output_stream = bos;
	    String command_output = Gatherer.remoteGreenstoneServer.runScript(collection_name, script_name, script_args, this);
	    this.commandOutput = command_output;
	    status = (command_output.equals("") ? CANCELLED : OK);
	}
	catch (Exception exception) {
	    DebugStream.printStackTrace(exception);
	    status = ERROR;
	}
    }


    // old_runLocal uses a potentially unsafe way of running a process and an inefficient way of reading
    // from the process stdout and stderr streams. Replaced by new runLocal() which uses SafeProcess and
    // the new LineByLineHandler inner class instantiated for each of the 2 process output streams.
    // If this method is ever removed from the file, can remove its helper get_stream_char() too.
    private void old_runLocal(String[] args, BufferedOutputStream bos) 
    {
	try {
	    String command = "";
	    for(int i = 0; i < args.length; i++) {
		command = command + args[i] + " ";
	    }
	    
	    ///ystem.err.println("Command: " + command);
	    fireMessage(type, Dictionary.get("GShell.Command") + ": " + command, status, null);
	    
	    Runtime rt = Runtime.getRuntime();
	    Process prcs = null;
	  
	    prcs = rt.exec(args);
	    // If we used single argument exec, spaces in filenames or paths cause problems
	    // I.e. better to avoid: prcs = rt.exec(command);
	    
	    InputStreamReader eisr = new InputStreamReader( prcs.getErrorStream(), "UTF-8" );
	    InputStreamReader stdisr = new InputStreamReader( prcs.getInputStream(), "UTF-8"  );	    
	
	    StringBuffer eline_buffer = new StringBuffer();
	    StringBuffer stdline_buffer = new StringBuffer();
	
	    while(/*type != GShell.NEW &&*/ SafeProcess.processRunning(prcs) && !hasSignalledStop()) {
		// Hopefully this doesn't block if the process is trying to write to STDOUT.
		if((eisr!=null) && eisr.ready()) { 
		    eline_buffer = get_stream_char(eisr,eline_buffer,bos);
		}
		// Hopefully this won't block if the process is trying to write to STDERR
		else if(stdisr.ready()) {
		    stdline_buffer = get_stream_char(stdisr,stdline_buffer,bos);
		}
		else {
		    try {
			sleep(100);
		    }
		    catch(Exception exception) {	     
		    }
		}
	    }
	
	    if(!hasSignalledStop()) {
		// Of course, just because the process is finished doesn't
		// mean the incoming streams are empty. Unfortunately I've
		// got no chance of preserving order, so I'll process the
		// error stream first, then the out stream
		while(eisr.ready()) {
		    eline_buffer = get_stream_char(eisr,eline_buffer,bos);
		}
	    
		while(stdisr.ready()) {
		    stdline_buffer = get_stream_char(stdisr,stdline_buffer,bos);
		}
	    
		// Ensure that any messages still remaining in the string buffers are fired off.
		if(eline_buffer.length() > 0) {
		    String eline = eline_buffer.toString();
		    //DebugStream.println("Last bit of eline: " + eline);
		    fireMessage(type, typeAsString(type) + "> " + eline, status, bos);
		    eline = null;
		}
	    
		if(stdline_buffer.length() > 0) {
		    String stdline = stdline_buffer.toString();
		    //DebugStream.println("Last bit of stdline: " + stdline);
		    fireMessage(type, typeAsString(type) + "> " + stdline, status, null);
		    stdline = null;
		}
	    }
	    else {
		DebugStream.println("We've been asked to stop.");
	    }
	
	
	    if(!hasSignalledStop()) {
		// Now display final message based on exit value
	    
		prcs.waitFor(); 

		if(prcs.exitValue() == 0) {
		    status = OK;
		    fireMessage(type, typeAsString(type) + "> " + Dictionary.get("GShell.Success"), status, null);
		}
		else {
		    status = ERROR;
		    fireMessage(type, typeAsString(type) + "> " + Dictionary.get("GShell.Failure"), status, null);
		}
	    
		eisr.close();
		stdisr.close();
	    }
	    else {
		// I need to somehow kill the child process. Unfortunately
		// Thread.stop() and Process.destroy() both fail to do
		// this. But now, thankx to the magic of Michaels 'close the
		// stream suggestion', it works fine (no it doesn't!)
		prcs.getInputStream().close();
		prcs.getErrorStream().close();
		prcs.getOutputStream().close();
		prcs.destroy();
		status = CANCELLED;
	    }
	}
	// Exception
	catch (Exception exception) {
	    DebugStream.println("Exception in GShell.runLocal() - unexpected");
	    DebugStream.printStackTrace(exception);
	    status = ERROR;
	}
    }
    
    private void runLocal(String[] args, BufferedOutputStream bos) 
    {
	// in case we stop between import and build, let's not launch further processes
	// that only pressing cancel, which results in interrupts, can subsequently stop.
	if(hasSignalledStop()) {
	    return;
	}

	String command = "";
	for(int i = 0; i < args.length; i++) {
	    command = command + args[i] + " ";
	}
	
	///ystem.err.println("Command: " + command);
	fireMessage(type, Dictionary.get("GShell.Command") + ": " + command, status, null);
	
	prcs = new SafeProcess(args);
	SafeProcess.LineByLineHandler processOutLineHandler
	    = new SynchronizedLineByLineHandler(SafeProcess.STDOUT);
	SafeProcess.LineByLineHandler processErrLineHandler
	    = new SynchronizedLineByLineHandler(SafeProcess.STDERR);
	
	prcs.setExceptionHandler(this); 
	int exitValue = prcs.runProcess(processOutLineHandler, processErrLineHandler); // use default procIn handling
	
	if(exitValue == 0) {
	    //System.err.println("*** Exitted normally");
	    status = OK;
	    fireMessage(type, typeAsString(type) + "> " + Dictionary.get("GShell.Success"), status, null);
	}
	else {
	    //System.err.println("*** Exitted abnormally with exit value " + exitValue);
	    status = ERROR;
	    fireMessage(type, typeAsString(type) + "> " + Dictionary.get("GShell.Failure"), status, null);
	}
	
	prcs = null;
    }

    /** When cancel is called on this GShell thread from a separate thread, 
     * This GShell will safely terminate any process it's currently running (by interrupting it)
     * and will set this.prcs to null at the end. */
    public void cancel() {
	if(prcs != null) { //GShell is running a process, so interrupt the GShell/SafeProcess thread 

	    SafeProcess.log("********** HAS SIGNALLED STOP. INTERRUPTING THE GSHELL/SAFEPROCESS THREAD");

	    // The following is implemented to interrupt the SafeProcess thread (which is *this* GShell thread),
		// but only if any process is being run by SafeProcess.
		// This will propagate the CANCEL status to any worker threads launched by the SafeProcess.
		// Everything will clean up nicely after itself (even terminating subprocesses that prcs launched on Windows!)
	    prcs.cancelRunningProcess(); 
	    
	    prcs = null;	    
	}
	setStatus(CANCELLED); //or: hasSignalledStop(); // synchronized. Either will set status to CANCELLED in thread-safe manner.	
    }
    

    /** Any threaded class must include this method to allow the thread body to be run. */
    public void run() {
	String col_name = args[args.length-1];		

	// Determine if the user has asked for an outfile.
	String out_name = null;
	BufferedOutputStream bos = null;
	if(type == IMPORT || type == BUILD || type == SCHEDULE) {
	    if(type == IMPORT) {
		out_name = (String) Gatherer.c_man.getCollection().import_options.getValue("out");
	    }
	    else if(type == BUILD) {
		out_name = (String) Gatherer.c_man.getCollection().build_options.getValue("out");
	    }
	    else { // SCHEDULE
		out_name = (String) Gatherer.c_man.getCollection().schedule_options.getValue("out"); 
	    }
	    if(out_name != null && out_name.length() > 0) {
		try {
		    bos = new BufferedOutputStream(new FileOutputStream(new File(out_name), true));
		}
		catch (Exception error) {
		    DebugStream.printStackTrace(error);
		}
	    }
	}

	// Issue a processBegun event
	//ystem.err.println("\nFiring process begun for " + type + "...");
	fireProcessBegun(type, status);
	//ystem.err.println("Done process begun.");
	if (Gatherer.isGsdlRemote) {
	    runRemote(args,bos);
	}
	else {
	    runLocal(args,bos);
	}
	//ystem.err.println("Done runLocal().");

	if(status == OK) {
	    if (type == NEW) {
		if (Gatherer.isGsdlRemote) {
		    Gatherer.remoteGreenstoneServer.downloadCollection(col_name);
		}
	    }
	    else if(type == IMPORT) {
		
		// download the archives directory (if gsdl server is remote)			
		if (Gatherer.isGsdlRemote) {

		    if (progress!=null) {
			progress.messageOnProgressBar("Downloading archive data from server");
		    }

		    Gatherer.remoteGreenstoneServer.downloadCollectionArchives(col_name);

		    if (progress!=null) {
			progress.messageOnProgressBar("");
		    }      
		}

		// Refresh the DocXMLFileManager
		fireMessage(type, typeAsString(type) + "> " + Dictionary.get("GShell.Parsing_Metadata_Start"), status, null);
		DocXMLFileManager.clearDocXMLFiles();
		if (Configuration.fedora_info != null && Configuration.fedora_info.isActive()) { // FLI case
		    File collection_export_directory = new File(CollectionManager.getLoadedCollectionExportDirectoryPath());
		    DocXMLFileManager.loadDocXMLFiles(collection_export_directory,"docmets.xml");
		}
		else {
		    File collection_archives_directory = new File(CollectionManager.getLoadedCollectionArchivesDirectoryPath());
		    DocXMLFileManager.loadDocXMLFiles(collection_archives_directory,"doc.xml");
		}


		fireMessage(type, typeAsString(type) + "> " + Dictionary.get("GShell.Parsing_Metadata_Complete"), status, null);
	    }

 	    else if(type == BUILD) {
 		// download the index (no longer building) directory if gsdl server is remote
 		if ((Gatherer.isGsdlRemote) && (!Configuration.fedora_info.isActive())) {
 		    if (progress!=null) {
 			progress.messageOnProgressBar("Downloading index data from server");
 		    }
		    			
		    if (!Gatherer.GS3){
			// Only need to download build.cfg file
			//File build_cfg_file = new File(CollectionManager.getLoadedCollectionBuildingDirectoryPath(), "build.cfg");
			File build_cfg_file = new File(CollectionManager.getLoadedCollectionIndexDirectoryPath(), "build.cfg");
			Gatherer.remoteGreenstoneServer.downloadCollectionFile(col_name, build_cfg_file);
		    }else{
			// Only need to download buildConfig.xml file
			//File buildConfig_xml_file = new File(CollectionManager.getLoadedCollectionBuildingDirectoryPath(), "buildConfig.xml");
			File buildConfig_xml_file = new File(CollectionManager.getLoadedCollectionIndexDirectoryPath(), "buildConfig.xml");
			Gatherer.remoteGreenstoneServer.downloadCollectionFile(col_name, buildConfig_xml_file);
		    }

 		    if (progress!=null) {
 			progress.messageOnProgressBar("");
 		    }
 		}
 	    }
	    else if(type == CDIMAGE) {
		// download exported files from tmp folder (if gsdl server is remote)			
		if (Gatherer.isGsdlRemote) {
		    if (progress!=null) {
			progress.messageOnProgressBar("Downloading CD-ROM data from server");
		    }

		    // !! TO DO
		    
 		    if (progress!=null) {
 			progress.messageOnProgressBar("");
 		    }
		}
	    }
	}

	// We're done.	
	//ystem.err.println("Firing process complete for " + type + "...");
	fireProcessComplete(type, status);
	// Close bos
	if(bos != null) {
	    try {
		bos.close();
		bos = null;
	    }
	    catch(Exception error) {
		DebugStream.printStackTrace(error);
	    }
	}
    }

    // Now synchronized, since separate threads handling process out and err streams can call fireMessage()
    // and need to reserve member objects and member variables used in this method while doing so.
    public synchronized void fireMessage(String message)
    {
	fireMessage(type, typeAsString(type) + "> " + message, status, buffered_output_stream);
    }


    /** Method for firing a message to all interested listeners.
     * @param type An <strong>int</strong> indicating the process type.
     * @param message The message as a <strong>String</strong>.
     * @param status An <strong>int</strong> specifying the current status of the process.
     */
    public void fireMessage(int type, String message, int status, BufferedOutputStream bos) {
	GShellEvent event = new GShellEvent(this, 0, type, message, status);
	// If there is a progress monitor attached, pass the event to it first. Note that we pass a queue of messages as the processing may cause one message to be split into several.
	ArrayList message_queue = new ArrayList();
	message_queue.add(event);
	if(progress != null) {
	    progress.process(message_queue);
	}
	for(int j = 0; j < message_queue.size(); j++) {
	    GShellEvent current_event = (GShellEvent) message_queue.get(j);
	    // If the event hasn't been vetoed, pass it on to other listeners
	    if(!current_event.isVetoed()) {
		// See http://stackoverflow.com/questions/8259479/should-i-synchronize-listener-notifications-or-not
		for (GShellListener l: this.listeners) {
		    l.message(current_event);
		}
	    }
	}
	// And if we have a buffered output stream from error messages, send the message there
	if(bos != null) {
	    try {
		bos.write(message.getBytes(), 0, message.length());
	    }
	    catch(Exception exception) {
		DebugStream.println("Exception in GShell.fireMessage() - unexpected");
		DebugStream.printStackTrace(exception);
	    }
	}
	message_queue = null;
	event = null;
    }

    /** Method for firing a process begun event which is called, strangely enough, when the process begins.
     * @param type An <strong>int</strong> indicating the process type.
     * @param status An <strong>int</strong> specifying the current status of the process.
     */
    protected void fireProcessBegun(int type, int status) {
	// Start the progres monitor if available
	if(progress != null) {
	    //ystem.err.println("About to call progress.start().");
	    progress.start();
	    //ystem.err.println("Called progress.start().");
	}
	// Fire an event
	GShellEvent event = new GShellEvent(this, 0, type, "", status);
	for (GShellListener l: this.listeners) {
	    l.processBegun(event);
	}
    }


    /** Method for firing a process complete event which is called, no surprise here, when the process ends.
     * @param type An <strong>int</strong> indicating the process type.
     * @param status An <strong>int</strong> specifying the current status of the process.
     */
    protected void fireProcessComplete(int type, int status) {
     Gatherer.invokeInEDT_replacesProceedInCurrThread("GShell.fireProcessComplete()", Gatherer.SYNC, new Runnable() {
      public void run() {

	// Tidy up by stopping the progress bar. If it was cancelled then the cancel command has arrived via the progress bars and they don't need to be told again (it actually causes problems).
	if(progress != null && status != CANCELLED) {
	    progress.stop();
	}
	
	// If we were cancelled, and we are lower details modes, fire off one last message.
	if(status == CANCELLED && Configuration.getMode() <= Configuration.LIBRARIAN_MODE) {
	    GShellEvent current_event = new GShellEvent(GShell.this, 0, type, Dictionary.get("GShell.Build.BuildCancelled"), status);
	    for (GShellListener l: GShell.this.listeners) {
		l.message(current_event);
	    }
	}
	
	String msg = "";
	// If we are creating collection and have trouble with permissions, we need more messages
	if(status == ERROR && type == GShell.NEW){
	  msg = args[args.length-1];
	}
	// And firing off an event
	GShellEvent event = new GShellEvent(GShell.this, 0, type, msg, status);
	for (GShellListener l: GShell.this.listeners) {
	    l.processComplete(event);
	}

      } // end run in EDT
     });	
    }

    /** Method to determine if the user, via the progress monitor, has signalled stop.
     * @return A <strong>boolean</strong> indicating if the user wanted to stop.
     */
    public synchronized boolean hasSignalledStop() {
	boolean has_signalled_stop = false;
	if(progress != null) {
	    has_signalled_stop = progress.hasSignalledStop();
	}
	if(has_signalled_stop) {
	    status = CANCELLED;
	}
	return has_signalled_stop;
    }

    /** Converts a type into a text representation.
     * @param type An <strong>int</strong> which maps to a shell process type.
     * @return A <strong>String</strong> which is the thread process's text name.
     */
    public String typeAsString(int type) {
	String name = null;
	switch(type) {
	case BUILD:
	    name = "buildcol.pl";
	    break;
	case IMPORT:
	    name = "import.pl";
	    break;
	case NEW:
	    name = "mkcol.pl";
	    break;
	case EXPORTAS:
	    name = "export.pl";
	    break;
	case CDIMAGE:
	    name = "exportcol.pl";
	    break;
	case CONVERT:
	    name = "convert_coll_from_gs2.pl";
	    break;
	case EXPLODE:
	    name = "explode_metadata_database.pl";
	    break;
	case SRCREPLACE: // To work with replace_srcdoc_with_html.pl
	    name = "replace_srcdoc_with_html.pl";
	    break;
	case SCHEDULE:
	    name = "schedule.pl";
	    break; 
	default:
	    name = "";
	}
	return name;
    }

    // From interface SafeProcess.ExceptionHandler
    // Called when an exception happens during the running of our perl process, as we want to set
    // the GShell status to ERROR.
    // However, exceptions when reading from our perl process' stderr and stdout streams are handled
    // by SynchronizedProcessHandler.gotException() below, since they happen in separate threads
    // from this one (the one from which the perl process is run).
    public void gotException(Exception e) {	

	if(e instanceof InterruptedException) {
	    DebugStream.println("We've been asked to stop.");
	    SafeProcess.log("@@@ Interruption to SafeProcess run by GShell.runLocal()");
	    setStatus(CANCELLED); // expected exception
	} else {
	    DebugStream.println("Exception in GShell.runLocal() - unexpected");
	    DebugStream.printStackTrace(e);
	    setStatus(ERROR); // status particularly needs to be synchronized on
	}
    }

    public synchronized void setStatus(int newStatus) {
	status = newStatus;
    }

    // Each instance of this class is run in its own thread by class SafeProcess.InputGobbler.
    // This class deals with each incoming line from the perl process' stderr or stdout streams. One
    // instance of this class for each stream. However, since multiple instances of this LineByLineHandler
    // are firing off events to the same outputstream object (bos) in their own threads, several objects,
    // not just bos, needed to be made threadsafe. So gotLine needs to be synchronized, and created a
    // synchronized setStatus() method above too.
    // This class also handles exceptions that may occur when reading from the perl process' stderr or stout.
    // The old GShell.runLocal() code would set the GShell.status to ERROR on each exception, so we ensure
    // we do that here too, to continue original behaviour. These calls are also synchronized to make their
    // use of the EventListeners threadsafe.
    protected class SynchronizedLineByLineHandler extends SafeProcess.LineByLineHandler
    {
	public SynchronizedLineByLineHandler(int src) {
	    super(src); // will set this.source to STDERR or STDOUT
	}

	// trying to keep synchronized methods as short as possible
	public void gotException(Exception e) {
	    String msg = "Got exception when processing the perl process' " + SafeProcess.streamToString(this.source) + " stream.";

	    DebugStream.println(msg); // method is already synchronized
	    DebugStream.printStackTrace(e); // method is already synchronized
	    SafeProcess.log("*** " + msg, e);

	    GShell.this.setStatus(GShell.ERROR); // synchronized
	}

	private void log(String msg) {
	    DebugStream.println(msg); // already synchro
	    System.err.println("@@@@@ " + msg);
	}

	// every time we read a line from the SafeProcess' stderr or stdout stream we come here.
	// Don't synchronize the gotLine() method, as it will synchronize on *this* (this object, this 
	// instance of LineByLineHandler). But we want to synchronize on the (outer class) GShell's
	// variables that are used in fireMessage(). So it's the fireMessage(String) method that
	// needs to be synchronized. See
	// http://stackoverflow.com/questions/574240/is-there-an-advantage-to-use-a-synchronized-method-instead-of-a-synchronized-blo
	public void gotLine(String line) {
	    fireMessage(line);     // synchronized, so even though process STDERR and STDOUT threads
	                           // will be firing messages, they won't be able to do so simultaneously
	}

    }

}
